%% ReviewPaper_Worldmap_v1.m

% Version 1 - 19 June 2024 - All OK.

% Roeland C. van de Vijsel
% Hydrology and Environmental Hydraulics Group
% Wageningen University and Research
% The Netherlands
% https://www.wur.nl/en/research-results/chair-groups/environmental-
% sciences/hydrology-and-environmental-hydraulics-group.htm
  
% This Matlab script creates a worldmap that shows the trends in major
% forcing factors (compound flood risk, sediment retention and relative
% sea-level rise ) for 47 of the world's major river deltas. 

% This worldmap is shown in the following publication:

% van de Vijsel, R.C., Scheffer, M. & Hoitink, A.J.F. (2024).
% Tipping points in river deltas. Nature Reviews Earth & Environment. 
 
% Whenever you use any of the contents of this dataset, please refer to
% this dataset as well as the associated publication.

clear all
close all
clc

% OK.

%% Add cmocean colormap functions

addpath cmocean\

% OK.

%% Choose how to determine correlations

% Choose how to calculate correlations between surge and discharge maxima within each year
CorrType = 'Kendall'; % 'Kendall'; % 'Spearman';

% OK.

%% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% 1. Calculate the trend in correlation between maximal river discharge and storm surge (Couasnon et al. 2020, NHESS)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Couasnon2020NHESS:
% Eilander, D. (2019). Paired time series of daily discharge and storm
% surge [Data set]. Zenodo. https://doi.org/10.5281/zenodo.3552820.

while(1)    % Create a while-loop, because halfway the script, a python-script first needs 
            % to be run with the intermediate output of this matlab script. Only once that is 
            % done, this matlab script should continue
   
    % Explore the input data
    
    datapath = 'Couasnon2020NHESS';
    addpath(datapath)
        
    % Explore the input data
    inputname = 'discharge_surge_reanalysis_v2.nc';
    ncdisp(inputname)
    
    % OK.
    
    %% Load data
    
    % time, in days since 1980-01-01
    time_1980 = ncread(inputname,'time');
    
    % (lat,lon) of GTSM measurement stations
    station_lat = ncread(inputname,'station_lat');
    station_lon = ncread(inputname,'station_lon');
    
    % (lat,lon) of the river mouths that are nearest to each GTSM station
    lat = ncread(inputname,'lat');
    lon = ncread(inputname,'lon');
    
    % Daily mean discharge at river mouth [m3/s]
    discharge = ncread(inputname,'discharge');
    
    % Daily max. surge (non-tidal residual) compoment of sea surface level [m]
    max_surge = ncread(inputname,'max_surge');
    
    % OK.
    
    %% Check that (lat,lon) and (station_lat,station_lon) are close together
    
    close(figure(1));figure(1)
    
    subplot(2,4,1)
    plot(lat,'or')
    hold on
    plot(station_lat,'.k')
    xlabel('station number')
    ylabel('lat')
    
    subplot(2,4,2)
    plot(lon,'or')
    hold on
    plot(station_lon,'.k')
    xlabel('station number')
    ylabel('lon')
    
    subplot(2,4,5)
    plot(lat-station_lat,'xc')
    xlabel('station number')
    ylabel('lat - station\_lat')
    
    subplot(2,4,6)
    plot(lon-station_lon,'xc')
    xlabel('station number')
    ylabel('lon - station\_lon')
    
    subplot(2,4,[3,4,7,8])
    plot(lon,lat,'or')
    hold on
    plot(station_lon,station_lat,'.k')
    xlabel('lon')
    ylabel('lat')
    
    % Maximal distance between river mouth and corresponding GTMS station is 75km
    rivmth_to_station_distance = ncread(inputname,'rivmth_to_station_distance');
    disp(['max distance is ', num2str(max(rivmth_to_station_distance)/1000),' km'])
    
    % OK.
    
    %% Check that distance between (lat,lon) and (station_lat,station_lon) is the same as the vector "rivmth_to_station_distance"
    
    wgs84 = wgs84Ellipsoid("m"); % Define reference ellipsoid
    dist = -999*ones(size(lat)); % Calculate distances between (lat,lon) and (station_lat,station_lon)
    for i = 1:length(dist) 
        dist(i) = distance(lat(i),lon(i),station_lat(i),station_lon(i),wgs84);
    end
    
    close(figure(2));figure(2) % Compare the two vectors. Conclusion: indeed the same.
    plot(dist,rivmth_to_station_distance,'or')
    hold on
    plot([0,8e4],[0,8e4],'k')
    xlabel('d [m]')
    ylabel(strrep('rivmth_to_station_distance [m]','_','\_'))
    
    % OK.
    
    %% Clear obsolete variables
    
    clear station_lat station_lon dist rivmth_to_station_distance
    
    % OK.
    
    %% Convert the time vector to datetime values
    
    dn = datenum("1980-01-01"); % The day number at the start of the time vector
    dd = -999*ones(size(time_1980)); % Create new array to store datetime values
    
    for i = 1:length(dd)
        dd(i) = addtodate(dn,time_1980(i),"day"); % Add the number of days from the time vector to get the datetime values
    end
    
    time = datetime(dd,"ConvertFrom","datenum"); % Datetime array
    
    % OK.
    
    %% Separate each element from the time vector into a cell for its specific year
    
    yy_vec = year(time(1)):year(time(end)); % All years in the time series
    yy_els = cell(size(yy_vec)); % Create a cell for each year
    
    for i = 1:length(time) % Loop over all elements from the time vector
    
        yy = year(time(i)); % The year of this specific element
        yi = yy-yy_vec(1) + 1; % The year index (1 to length(yy_vec))
    
        yy_els{yi} = [yy_els{yi},i]; % Assign the index of this element to the separate cell for this year
        
        disp(i)
    
    end
    
    % OK.
    
    %% Find the co-occurring maximum surges and discharges over the total time period, for each river mouth and calculate correlations
    
    rivmths_NumCooc = -999*ones(size(lat)); % Create an array to store the total number of co-occurring surge and discharge maxima for each river mouth
    
    rho_arr  = -999*ones(length(yy_vec),length(lat)); % Create new array to store Spearman/Kendall correlation coeff, for every year, for every river mouth
    pval_arr = -999*ones(length(yy_vec),length(lat)); % Create new array to store corresponding p-value, for every year, for every river mouth
    
    for rmi = 1:length(lat) % loop over all river mouth ids
    
        NumCooc = 0; % For each river mouth, set the counter to zero at the start of the calculation loop
    
        for yi = 1:length(yy_vec) % loop over all years
    
            % Find the co-occurring maximum surges and discharges over the total time period, for each river mouth
    
            max_surge_yy = max_surge(yy_els{yi},rmi); % Identify the surge values for this year, for this river mouth
            max_surge_el = find(max_surge_yy == max(max_surge_yy)); % Find the element of the maximum surge value of this year
            max_surge_time = time(yy_els{yi}(max_surge_el)); % Find the corresponding time
    
            discharge_yy = discharge(yy_els{yi},rmi); % Identify the discharge values for this year, for this river mouth
            discharge_el = find(discharge_yy == max(discharge_yy)); % Find the element of the maximum discharge value of this year
            discharge_time = time(yy_els{yi}(discharge_el)); % Find the corresponding time
            
            dist_days = days(discharge_time - max_surge_time); % Find the number of days between the maximum surge and discharge
    
            if abs(dist_days) <= 3
                NumCooc = NumCooc + 1; % If discharge and surge maxima are within 3 days of each other, count this event as co-occurring maxima
            end
    
            % Calculate Spearman/Kendall correlation coefficient between daily surge and discharge maxima, for every year, for every river mouth
            
            nan_el = (isnan(max_surge_yy) | isnan(discharge_yy)); % Find all elements where either surge or discharge have a NaN
            X = max_surge_yy(~nan_el);                  % Surge data, with all NaNs discarded
            Y = discharge_yy(~nan_el);                  % Discharge data, with all NaNs discarded
    
            [rho,pval] = corr(X,Y,'Type',CorrType);     % Calculate correlation coefficient and corresponding p-value
            rho_arr(yi,rmi) = rho;                      % Store in arrays
            pval_arr(yi,rmi) = pval;                    % Store in arrays
    
            clc; disp(['yi: ',num2str(100*yi/length(yy_vec)),'%; rmi: ',num2str(100*rmi/length(lat)),'%']) % Show the progress
    
        end
    
        rivmths_NumCooc(rmi) = NumCooc; % Save the total count for this river mouth
    
    end
    
    % OK.
    
    
    %% Write rho_arr to a CSV-file, to load it in Python to compute the Mann-Kendall test statistic
    
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    
    writematrix(rho_arr,'MannKendall/rho_arr.csv','Delimiter',';')
    
    % OK.
    
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

    choice = menu('Run the Python script "MannKendall/MannKendall.ipynb". After that, press CONTINUE','CONTINUE');
    
    if choice==1
       break;
    end

end % Once the python script has been executed, this matlab script can continue

% OK.

%% Import the Mann-Kendall test results from the Python script (MannKendall.ipynb)

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

MannKendall.h_dt         = importdata('MannKendall/h_dt.txt');
MannKendall.p_dt         = importdata('MannKendall/p_dt.txt');
MannKendall.z_dt         = importdata('MannKendall/z_dt.txt');
MannKendall.Tau_dt       = importdata('MannKendall/Tau_dt.txt');
MannKendall.s_dt         = importdata('MannKendall/s_dt.txt');
MannKendall.var_s_dt     = importdata('MannKendall/var_s_dt.txt');
MannKendall.slope_dt     = importdata('MannKendall/slope_dt.txt');
MannKendall.intercept_dt = importdata('MannKendall/intercept_dt.txt');

% OK.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Use these results to save as rho_dt and pval_dt

rho_dt = MannKendall.slope_dt; % The trend in the correlation coefficient [yr^-1], for each river mouth
pval_dt = MannKendall.p_dt;    % Associated p-value of this trend

% OK.

%% Set lat and lon limits for all world maps

latlim = [-60 85];
lonlim = [-180 180];

% OK.

%% Plot global distribution of the number of co-occurring surge & discharge maxima over 1980 - 2014

close(figure(3)); figure(3)

ax = worldmap(latlim,lonlim);
% ax = worldmap('World');
setm(ax,'mapprojection','eqdcylin');
land = shaperead('landareas.shp', 'UseGeoCoords', true);
geoshow(ax, land, 'FaceColor', [0.9 0.9 0.9])
hold on
    scatterm(lat,lon,10,rivmths_NumCooc,'Filled')
% geoscatter(lat,lon,10,rivmths_NumCooc,'Filled')

cmap = flipud(cmocean('ice'));
colormap(cmap)
cb = colorbar;
title('Nr. co-occurring annual discharge (Q) + surge (S) maxima (1980-2014)')

plabel off
mlabel off

cbp = get(cb,'Position'); % [left, bottom, width, height]
set(cb,'Position',[1.05*cbp(1) 1.195*cbp(2) cbp(3) 0.91*cbp(4)])

set(gcf,'Units','Inches');
pos = get(gcf,'Position');
set(gcf,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(gcf,['output','/','num_coocurring_discharge_surge.pdf'],'-dpdf','-r500')
print(gcf,['output','/','num_coocurring_discharge_surge.png'],'-dpng','-r500')

% OK.

%% Plot the Mann-Kendall trend in the Spearman/Kendall correlation [year^-1]

close(figure(4)); figure(4)

ax = worldmap(latlim,lonlim);
% ax = worldmap('World');
setm(ax,'mapprojection','eqdcylin');
land = shaperead('landareas.shp', 'UseGeoCoords', true);
geoshow(ax, land, 'FaceColor', [0.9 0.9 0.9])
hold on
    scatterm(lat,lon,10,rho_dt,'Filled')
% geoscatter(lat,lon,10,rho_dt,'Filled')

cmap = (cmocean('balance','pivot',0));
colormap(cmap)
cb = colorbar;
title('Trend in correlation cofficient, \partial\rho(Q,S)/\partialt [yr^{-1}]')

plabel off
mlabel off
cbp = get(cb,'Position'); % [left, bottom, width, height]
set(cb,'Position',[1.05*cbp(1) 1.195*cbp(2) cbp(3) 0.91*cbp(4)])

set(gcf,'Units','Inches');
pos = get(gcf,'Position');
set(gcf,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(gcf,['output','/','trend_corr_discharge_surge.pdf'],'-dpdf','-r500')
print(gcf,['output','/','trend_corr_discharge_surge.png'],'-dpng','-r500')

% OK.

%% Plot the corresponding p-value of the Mann-Kendall trend in the correlation coefficients

close(figure(5)); figure(5)

ax = worldmap(latlim,lonlim);
% ax = worldmap('World');
setm(ax,'mapprojection','eqdcylin');
land = shaperead('landareas.shp', 'UseGeoCoords', true);
geoshow(ax, land, 'FaceColor', [0.9 0.9 0.9])
hold on 
    scatterm(lat,lon,10,pval_dt,'Filled')
% geoscatter(lat,lon,10,pval_dt,'Filled')

cmap = flipud(cmocean('amp'));
colormap(cmap)
cb = colorbar;
title('p-value of the trend in correlation coefficient')

plabel off
mlabel off
cbp = get(cb,'Position'); % [left, bottom, width, height]
set(cb,'Position',[1.05*cbp(1) 1.195*cbp(2) cbp(3) 0.91*cbp(4)])

set(gcf,'Units','Inches');
pos = get(gcf,'Position');
set(gcf,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(gcf,['output','/','pval_trend_corr_discharge_surge.pdf'],'-dpdf','-r500')
print(gcf,['output','/','pval_trend_corr_discharge_surge.png'],'-dpng','-r500')

% OK.

%% Plot the Mann-Kendall trend in the Spearman/Kendall correlation [year^-1], but only for p-valua < 0.05

close(figure(6)); figure(6)

el_05 = find(pval_dt < 0.05); % Select only the river mouths where the trend in corr. had p-value < 0.05

ax = worldmap(latlim,lonlim);
% ax = worldmap('World');
setm(ax,'mapprojection','eqdcylin');
land = shaperead('landareas.shp', 'UseGeoCoords', true);
geoshow(ax, land, 'FaceColor', [0.9 0.9 0.9])
hold on
    scatterm(lat(el_05),lon(el_05),10,rho_dt(el_05),'Filled')
% geoscatter(lat(el_05),lon(el_05),10,rho_dt(el_05),'Filled')

cmap = (cmocean('balance','pivot',0));
colormap(cmap)
cb = colorbar;
title('Trend in correlation coefficient, \partial\rho(Q,S)/\partialt [yr^{-1}], for p < 0.05')

plabel off
mlabel off
cbp = get(cb,'Position'); % [left, bottom, width, height]
set(cb,'Position',[1.05*cbp(1) 1.195*cbp(2) cbp(3) 0.91*cbp(4)])

set(gcf,'Units','Inches');
pos = get(gcf,'Position');
set(gcf,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(gcf,['output','/','trend_corr_discharge_surge_p05.pdf'],'-dpdf','-r500')
print(gcf,['output','/','trend_corr_discharge_surge_p05.png'],'-dpng','-r500')

% OK.

%% Clear all variables, except those needed for subsequent calculations, and save

Couasnon2020NHESS.time   = time; % Time array of the discharge and surge values (daily datetimes, from 1980 to 2014)
Couasnon2020NHESS.yy_vec = yy_vec; % All years (= 1980, 1981, ..., 2014)

Couasnon2020NHESS.lat    = lat; % Latitude of all river mouths
Couasnon2020NHESS.lon    = lon; % Longitude of all river mouths
Couasnon2020NHESS.wgs84  = wgs84; % Coordinate system (WGS84)

Couasnon2020NHESS.discharge = discharge; % Daily discharge values (at times indicated in the vector "time")
Couasnon2020NHESS.max_surge = max_surge; % Daily maximal storm suge values (at times indicated in the vector "time")

Couasnon2020NHESS.rivmths_NumCooc = rivmths_NumCooc; % Total number of co-occurring annual maxima of discharge and surge, summed over 1980-2014, for each river mouth

Couasnon2020NHESS.rho_arr  = rho_arr;  % Spearman/Kendall correlation coefficient between daily timeseries of discharge and surge of each year separately, for each river mouth
Couasnon2020NHESS.pval_arr = pval_arr; % Corresponding p-value of this rank correlation coefficient

Couasnon2020NHESS.rho_dt  = rho_dt;  % Mann-Kendall trend in the Spearman/Kendall correlation coefficients as a function of time in years
Couasnon2020NHESS.pval_dt = pval_dt; % p-value of this trend

clearvars -except Couasnon2020NHESS latlim lonlim CorrType

clc

% OK.

%% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% 2. Calculate the relative sea-level change (Nienhuis et al. 2023, Annual Review of Earth and Planetary Sciences)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% GlobalDeltaData (v3.0):
% Jaap Nienhuis. (2022). jhnienhuis/GlobalDeltaChange: GlobalDeltaData v3.0 (3.0). Zenodo. https://doi.org/10.5281/zenodo.7044707

% GlobalDeltaSeaLevel (v2.0):
% https://github.com/jhnienhuis/GlobalDeltaSeaLevel/releases/tag/v2.0

%% Explore the input data

datapath = 'Nienhuis2023AREPS';
datapathGDSL = [datapath,'/','GlobalDeltaSeaLevel-2.0/GlobalDeltaSeaLevel-2.0'];
datapathGDC  = [datapath,'/','GlobalDeltaChange-3.0','/','jhnienhuis-GlobalDeltaChange-2be9f2f'];
addpath(datapath)
addpath(datapathGDSL)
addpath(datapathGDC)

% OK.

%% Load SLR data used for Fig. 5c in Nienhuis et al. (2023, Annual Review of Earth and Planetary Sciences)
% Here I followed the same steps as in the original script by J.H.
% Nienhuis, i.e. datapathGDC/pub_figures/annualreviews2023/Fig5_SLmaps.m

GlobalDeltaData = load([datapathGDC,'/','GlobalDeltaData.mat']); %,'MouthLon','MouthLat','QRiver_prist','Continent')

GlobalDeltaArea = load([datapathGDSL,'/','export_data/GlobalDeltaArea.mat']); %,'delta_area')

GlobalDeltaSeaLevelData = load([datapathGDSL,'/','export_data/GlobalDeltaSeaLevelData.mat']);

idx1 = GlobalDeltaArea.delta_area>1e7; % Select only deltas whose area is large enough

SLR = mean(1000*(GlobalDeltaSeaLevelData.DeltaSLR_SSP245(:,1:2)),2); % Projected SLR [mm/yr] (for 2000-2100, SSP2-4.5 scenario)

idx2 = ~isnan(SLR); % Filter out deltas whose SLR is not an NaN

idx = (idx1 & idx2); % Select deltas whose area is large enough AND whose SLR is not an NaN

% OK.

%% Check that the 3 data structures are ordened in the same way

BasinID1 = GlobalDeltaData.BasinID2;
BasinID2 = GlobalDeltaArea.BasinID2;
BasinID3 = GlobalDeltaSeaLevelData.BasinID2;

% Basin IDs are all the same

all(BasinID1 == BasinID2)
all(BasinID1 == BasinID3)

% lat,lon: lat1,lon1 = lat3,lon3, but lat2,lon2 is different

lat1 = GlobalDeltaData.MouthLat;
lon1 = GlobalDeltaData.MouthLon;

lat2 = GlobalDeltaArea.mouth_latlon(:,1);
lon2 = GlobalDeltaArea.mouth_latlon(:,2);

lat3 = GlobalDeltaSeaLevelData.MouthLat;
lon3 = GlobalDeltaSeaLevelData.MouthLon;

%

all(lat1 == lat3)
all(lon1 == lon3)

% Indeed: the delta names of set 1 and 3 are the same (set 2 has no names)

all(GlobalDeltaData.delta_name == GlobalDeltaSeaLevelData.delta_name)

% Quantify the difference (distances) between set 1/3 and set 2

dist12 = -999*ones(size(lat1));
dist13 = -999*ones(size(lat1));
dist23 = -999*ones(size(lat1));

wgs84 = wgs84Ellipsoid("kilometer"); % Define reference ellipsoid

for i = 1:length(lat1)
    dist12(i) = distance(lat1(i),lon1(i),lat2(i),lon2(i),wgs84);
    dist13(i) = distance(lat1(i),lon1(i),lat3(i),lon3(i),wgs84);
    dist23(i) = distance(lat2(i),lon2(i),lat3(i),lon3(i),wgs84);
    disp(i)
end

% Plot distances

close(figure(7));figure(7)

subplot(2,1,1)
plot(dist12,'.-b')
ylabel('distance [km]')
xlabel('river mouth')

subplot(2,1,2)
scatter(GlobalDeltaArea.delta_area,dist12)
ylabel('distance [km]')
xlabel('delta area')

% Conclusion: the variable GlobalDeltaArea.delta_area should be used with
% care (but in the remainder of this script, we don't really use it anyway.
% Only to select deltas whose area are relatively large (for plotting
% purposes). But for the final analyses, we have selected ca. 47 deltas
% that are large anyway (so we don't need GlobalDeltaArea.delta_area).

% OK.

%% Plot these SLR projections in a world map

close(figure(8)); figure(8)

ax = worldmap(latlim,lonlim);
% ax = worldmap('World');
setm(ax,'mapprojection','eqdcylin');
land = shaperead('landareas.shp', 'UseGeoCoords', true);
geoshow(ax, land, 'FaceColor', [0.9 0.9 0.9])
hold on
    scatterm(GlobalDeltaData.MouthLat(idx),GlobalDeltaData.MouthLon(idx),10,SLR(idx),'Filled')
% geoscatter(MouthLat(idx),MouthLon(idx),10,SLR(idx),'Filled')

cmap = (cmocean('balance','pivot',0));
colormap(cmap)
cb = colorbar;

title('SLR [mm/yr] (2000-2100, SSP2-4.5); delta area > 10^7')

plabel off
mlabel off
cbp = get(cb,'Position'); % [left, bottom, width, height]
set(cb,'Position',[1.05*cbp(1) 1.195*cbp(2) cbp(3) 0.91*cbp(4)])

set(gcf,'Units','Inches');
pos = get(gcf,'Position');
set(gcf,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(gcf,['output','/','SLR_2000-2100_SSP2-45.pdf'],'-dpdf','-r500')
print(gcf,['output','/','SLR_2000-2100_SSP2-45.png'],'-dpng','-r500')

% OK.

%% Clear all variables, except those needed for subsequent calculations, and save

Nienhuis2023AREPS.MouthLat     = GlobalDeltaData.MouthLat;     % River mouth latitude
Nienhuis2023AREPS.MouthLon     = GlobalDeltaData.MouthLon;     % River mouth longitude
Nienhuis2023AREPS.delta_name   = GlobalDeltaData.delta_name;   % Delta name
Nienhuis2023AREPS.QRiver_prist = GlobalDeltaData.QRiver_prist; % Pristine riverine sediment influx
Nienhuis2023AREPS.QRiver_dist  = GlobalDeltaData.QRiver_dist;  % Disturbed riverine sediment influx
Nienhuis2023AREPS.SLR          = SLR;
Nienhuis2023AREPS.Continent    = GlobalDeltaData.Continent; 

clearvars -except Couasnon2020NHESS Nienhuis2023AREPS latlim lonlim CorrType

% OK.


%% Selection the ~47 major river deltas in the world (Dunn 2017)

% Dunn, F. E. (2017). Multidecadal fluvial sediment fluxes to major deltas
% under environmental change scenarios: Projections and their implications
% (Doctoral dissertation, University of Southampton).  

% The 47 major deltas are shown in Figure 2.3

% For each of these 47 deltas, I found the corresponding delta names from
% Nienhuis2023AREPS.delta_name. The list of these 47 delta names is given
% in "choose_deltas/selected_deltas.txt".

selected_deltas = readlines('selected_deltas.txt');

% OK.

%% Loop over all selected deltas to find their respective BasinID

selID = -999*ones(length(selected_deltas),1); % Create an array where the indices from the total Nienhuis2023AREPS-delta array are stored for the selected deltas
clc

for i = 1:length(selected_deltas)
   
    selid = find(selected_deltas{i} == Nienhuis2023AREPS.delta_name); % Find the matching index

    if length(selid) ~= 1
        error('More or less than one matching river delta found')
    else
        selID(i) = selid;
    end

    disp([num2str(i),': ', selected_deltas{i},'     =     ',Nienhuis2023AREPS.delta_name{selid}])

end

Nienhuis2023AREPS.selID = selID;
Nienhuis2023AREPS.selected_deltas = selected_deltas;

% OK.

%% Create a new structure, "delta", that stores only the info of the selected deltas

delta.lat           = Nienhuis2023AREPS.MouthLat(selID);
delta.lon           = Nienhuis2023AREPS.MouthLon(selID);
delta.name          = Nienhuis2023AREPS.delta_name(selID);
delta.QRiver_dist   = Nienhuis2023AREPS.QRiver_dist(selID);
delta.QRiver_prist  = Nienhuis2023AREPS.QRiver_prist(selID);
delta.SLR           = Nienhuis2023AREPS.SLR(selID);
delta.Continent     = Nienhuis2023AREPS.Continent(selID);

% Optimize the name strings for in the plots
delta.plot_name     = delta.name;
delta.plot_name{4}  = 'Chao Phraya';
delta.plot_name{5}  = 'Colorado';
% delta.plot_name{9}  = 'GBM';
delta.plot_name{24} = 'Murray';
delta.plot_name{27} = 'Han';
delta.plot_name{33} = 'Rio Grande';
delta.plot_name{37} = 'Red';
delta.plot_name{42} = 'Tone';
delta.plot_name{47} = 'Pearl';

% OK.

%% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% 3. Link the selected deltas from Nienhuis2023AREPS to the closest river
% mouths from Couasnon2020NHESS
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Find the corresponding lat/lon combis between the selected deltas and all the river mouths from Cuasnon2020NHESS

% Add 360 degrees to negative longitudes and subtract 360 degrees from
% longitudes larger than 360
Couasnon2020NHESS.lon(Couasnon2020NHESS.lon<0)   = Couasnon2020NHESS.lon(Couasnon2020NHESS.lon<0)   + 360;
Couasnon2020NHESS.lon(Couasnon2020NHESS.lon>360) = Couasnon2020NHESS.lon(Couasnon2020NHESS.lon>360) - 360;
delta.lon(delta.lon<0)   = delta.lon(delta.lon<0)   + 360;
delta.lon(delta.lon>360) = delta.lon(delta.lon>360) - 360;

min_els   = zeros(size(delta.name)); % Make an array that stores the indices where the distance between the selected deltas and Couasnon is minimal
min_dist  = zeros(size(delta.name)); % Make an array that stores the minimal distances between the selected deltas and Couasnon

wgs84 = wgs84Ellipsoid("kilometer"); % Define reference ellipsoid

for i = 1:length(delta.lat)

    % The (lat,lon), running through the list of selected deltas
    latS = delta.lat(i);
    lonS = delta.lon(i);

    % The entire array of (lat,lon) form the Couasnon dataset
    latC = Couasnon2020NHESS.lat;
    lonC = Couasnon2020NHESS.lon;

    dist = -999*ones(size(latC)); % Array to store distances between Nienhuis and Couasnon delta coordinates

    for j = 1:length(latC) % Calculate distances from the specified Nienhuis-delta to every Couasnon-delta
        dist(j) = abs(distance(latS,lonS,latC(j),lonC(j),wgs84));
    end

    min_el = find(dist == min(dist)); % The element in the Couasnon array where the distance is minimal

    if length(min_el) > 1
        warning('more than one matches')
        min_el = min_el(1);
    elseif length(min_el) == 0
        warning('no matches')
        min_el = NaN;
    end

    min_els(i) = min_el; % Save in array
    min_dist(i) = dist(min_el); % Save the actual distance between the Couasnon and Nienhuis river mouth (in km) in an array

    clc; disp([num2str(i),'/',num2str(length(delta.lat))])

end

% OK.

%% Add the selected Couasnon data to the "delta" structure

% Trend in surge-discharge correlation
delta.rho_dt  = Couasnon2020NHESS.rho_dt(min_els);  % Mann-Kendall trend in the Spearman/Kendall correlation coefficients as a function of time in years
delta.pval_dt = Couasnon2020NHESS.pval_dt(min_els); % p-value of this fitted trend

% Sediment retention
delta.sed     = (delta.QRiver_prist - delta.QRiver_dist)./delta.QRiver_prist; % Riverine sediment retention: influx under (pristine - disturbed)/pristine conditions (>0 = influx decrease; <0 = influx increase)

% Divide each variable by the maximal value, to normalize (such all values reach max 1)
delta.rho_dt_norm = delta.rho_dt/max(delta.rho_dt);
delta.sed_norm    = delta.sed/max(delta.sed);
delta.SLR_norm    = delta.SLR/max(delta.SLR);

% Idem, but divide each variable by the maximal ABSOLUTE value, to normalize (such all values reach max 1 or min -1)

delta.rho_dt_norm_abs = delta.rho_dt/max(abs(delta.rho_dt));
delta.sed_norm_abs    = delta.sed/max(abs(delta.sed));
delta.SLR_norm_abs    = delta.SLR/max(abs(delta.SLR));

% OK.

%% Check if all deltas are correctly identified and correspond (well enough) between Nienhuis & Couasnon

for N = 1:6 % Plot number

    close(figure(8+N)); figure(8+N)
    set(gcf,'units','normalized','outerposition',[0 0 1 1])
    
    dlat = 0.6; % latitude margin (degrees) around the selected delta
    dlon = 0.6; % longitude margin (degrees) around the selected delta
    
    
    for i = (1:9)+(N-1)*9 % Loop over all selected deltas (divided over N figures with 9 subplots)
        if i <= length(delta.lat)

            gx = subplot(3,3,i-(N-1)*9);
        
            % Show all river mouths from Couasnon (black filled circles)
            geoscatter(Couasnon2020NHESS.lat,Couasnon2020NHESS.lon,100,'k','Filled')
    
            hold on
    
            % Show the selected river mouth from Couasnon (cyan cross)
            geoscatter(Couasnon2020NHESS.lat(min_els),Couasnon2020NHESS.lon(min_els),100,'cx','LineWidth',2)
            
            % Show the selected river mouth from Nienhuis (red square)
            geoscatter(delta.lat(i),delta.lon(i),100,'rs','LineWidth',2)
        
            geobasemap topographic
        
            % Zoom in on this delta
            geolimits([delta.lat(i) - dlat, delta.lat(i) + dlat],[delta.lon(i) - dlon, delta.lon(i) + dlon])
        
            title([num2str(i),': ',delta.plot_name{i},': ',num2str(min_dist(i),'%.1f'),' km'])
        
        end
    end

end

% Conclusion:
% Some river deltas are in a slightly different location than the selected
% river mouth from Couasnon2020. But all selected river mouths are the
% closest-by to the selected delta, and they all are reasonably well
% representing the delta of choice. So all OK.

% OK.

%% Show bar graphs of combined threats, for all selected deltas

% Determine the y-axis range
barmin = min([min(delta.sed_norm),min(delta.rho_dt_norm),min(delta.SLR_norm)]);
barmax = max([max(delta.sed_norm),max(delta.rho_dt_norm),max(delta.SLR_norm)]);

% Choose a color scheme
cmap = flipud(cmocean('matter'));
elc = linspace(1,255,3);
col1 = cmap(elc(1),:);
col2 = cmap(elc(2),:);
col3 = cmap(elc(3),:);

fs = 12; % FontSize

close(figure(15)); figure(15)
set(gcf,'units','normalized','outerposition',[0 0 1 1])

for i = 1:length(delta.lat)
   
    subplot(4,12,i)
    
        h(1) = bar(0.5,delta.sed_norm(i),    'FaceColor',col1,'EdgeColor',[0 0 0]); hold on
    if delta.pval_dt(i) < 0.05 % Distinguish between statistically significant and non-significant trends
        h(2) = bar(1.5,delta.rho_dt_norm(i), 'FaceColor',col2,'EdgeColor',[0 0 0]);
    else
        h(2) = bar(1.5,delta.rho_dt_norm(i), 'FaceColor',col2,'EdgeColor',[0 0 0],'FaceAlpha',0.1); % For non-significant trends, increase bar transparency
    end
        h(3) = bar(2.5,delta.SLR_norm(i),    'FaceColor',col3,'EdgeColor',[0 0 0]);

    title(delta.plot_name(i))

    axis([0 3 1.05*barmin 1.05*barmax])
    set(gca,'XTickLabels',[],'YTickLabels',[],'FontSize',fs-3)

    if i == 23
        legend('Fluvial sediment retention [%]','Trend in surge-discharge correlation [yr^{-1}]','Relative sea-level change [mm yr^{-1}]','Position', [0.13 0.05 0.775 0.04],'NumColumns',3,'FontSize',fs-3,'FontWeight','bold')
    end

end

% plot a legend

subplot(4,12,4*12)

    h(1) = bar(0.5,max(delta.sed_norm),    'FaceColor',col1,'EdgeColor',[0 0 0],'FaceAlpha',0.5); hold on
    h(1) = bar(0.5,min(delta.sed_norm),    'FaceColor',col1,'EdgeColor',[0 0 0],'FaceAlpha',1.0);

    h(2) = bar(1.5,max(delta.rho_dt_norm), 'FaceColor',col2,'EdgeColor',[0 0 0],'FaceAlpha',0.5);
    h(2) = bar(1.5,min(delta.rho_dt_norm), 'FaceColor',col2,'EdgeColor',[0 0 0],'FaceAlpha',1.0);

    h(3) = bar(2.5,max(delta.SLR_norm),    'FaceColor',col3,'EdgeColor',[0 0 0],'FaceAlpha',0.5);
    h(3) = bar(2.5,min(delta.SLR_norm),    'FaceColor',col3,'EdgeColor',[0 0 0],'FaceAlpha',1.0);

title(delta.plot_name(i))

axis([0 3 1.05*barmin 1.05*barmax])
set(gca,'XTickLabels',[],'YTickLabels',[],'FontSize',fs-3)

addpath output

set(gcf,'Units','Inches');
pos = get(gcf,'Position');
set(gcf,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(gcf,['output/Threats_combined_bars.pdf'],'-dpdf','-r500')
print(gcf,['output/Threats_combined_bars.png'],'-dpng','-r500')

% OK.

%% Show bar graphs of combined threats, for all selected deltas (cmap = BALANCE)

% Determine the y-axis range
barmin = min([min(delta.sed_norm),min(delta.rho_dt_norm),min(delta.SLR_norm)]);
barmax = max([max(delta.sed_norm),max(delta.rho_dt_norm),max(delta.SLR_norm)]);

% Choose a color scheme and prepare to color the bars according to value
cmap = cmocean('balance'); % (blue <0, white 0, red >0)
cmap_amp = max([abs(barmin),abs(barmax)]); % Maximal absolute bar value
cmap_val = linspace(-cmap_amp,cmap_amp,length(cmap)); % Bar values that each of the 256 elements in the cmap corresponds to

fs = 12; % FontSize

close(figure(16)); figure(16)
set(gcf,'units','normalized','outerposition',[0 0 1 1])

for i = 1:length(delta.lat) % Loop over all selected deltas
   
    subplot(4,12,i)
    
    % For each of the 3 factors (sediment retention, flood correlation trend, RSL), determine the
    % color that should be used to fill the respective bar

    % Sediment retention
    val = delta.sed_norm(i); % Value of this bar
    cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val))); % Find the index from the colormap that corresponds to this value
    col1 = mean(cmap(cmap_ind,:),1); % Corresponding color (average in the "1"-direction, in case there are more than one cmap_ind's

    % Idem for flood correlation trend
    val = delta.rho_dt_norm(i);
    cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val)));
    col2 = mean(cmap(cmap_ind,:),1);

    % Idem for RSL
    val = delta.SLR_norm(i);
    cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val)));
    col3 = mean(cmap(cmap_ind,:),1);

    % Plot the bars

        h(1) = bar(0.5,delta.sed_norm(i),      'FaceColor',col1,  'EdgeColor',[0 0 0]); hold on

    if delta.pval_dt(i) < 0.05 % If p<0.05, fill the flood trend bar fully with colors

        h(2) = bar(1.5,delta.rho_dt_norm(i),   'FaceColor',col2,  'EdgeColor',[0 0 0]);

    else % For p-value >= 0.05: distinguish between 0.05<=p<0.10 and p>=0.10

        if (delta.pval_dt(i) >= 0.05) && (delta.pval_dt(i) < 0.10) % For "weakly non-significant" trends: divide the bar into 6 almost touching bars
            nb = 6; % number of (almost touching) bars
        elseif (delta.pval_dt(i) >= 0.10) % For "strongly non-significant" trends: divide the bar into 4 bars with quite some space in between
            nb = 4; % number of separated bars
        else
            error('something wrong with p-values')
        end

        w = 0.1; % relative width of the individual bars
        bc = linspace(3*w/2, 1-3*w/2, nb); % individual bar center locations
        for nbi = 1:nb % Draw each individual bar
            bci = 1.0+bc(nbi); % Define the center of the currently drawn individual bar
            h(2) = bar(bci,delta.rho_dt_norm(i),w, 'FaceColor',col2,  'EdgeColor','none');
        end
        h(2) = bar(1.5,delta.rho_dt_norm(i),   'FaceColor','none','EdgeColor',[0 0 0]); % Surround the group of individual thin bars with a black edge

    end

    h(3) = bar(2.5,delta.SLR_norm(i),      'FaceColor',col3,  'EdgeColor',[0 0 0]);

    title(delta.plot_name(i))

    axis([0 3 1.05*barmin 1.05*barmax])
    set(gca,'XTickLabels',[],'YTickLabels',[],'FontSize',fs-3)

    if i == 23
        legend('Fluvial sediment retention [%]','Trend in surge-discharge correlation [yr^{-1}]','Relative sea-level change [mm yr^{-1}]','Position', [0.13 0.05 0.775 0.04],'NumColumns',3,'FontSize',fs-3,'FontWeight','bold')
    end

end

% plot a legend

subplot(4,12,4*12)

val = max(delta.sed_norm);
leg_txt = num2str(100*max(delta.sed),'%.1f'); % Express in % in the legend
cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val))); % Index from the colormap
col1 = mean(cmap(cmap_ind,:),1); % Corresponding color
h(1) = bar(0.5,val,'FaceColor',col1,'EdgeColor',[0 0 0]); hold on
text(0.5,1.2*val,leg_txt,'HorizontalAlignment','center','FontSize',fs-3)

val = min(delta.sed_norm);
leg_txt = num2str(100*min(delta.sed),'%.1f'); % Express in % in the legend
cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val))); % Index from the colormap
col1 = mean(cmap(cmap_ind,:),1); % Corresponding color
h(1) = bar(0.5,val,'FaceColor',col1,'EdgeColor',[0 0 0]);
text(0.5,1.2*val,leg_txt,'HorizontalAlignment','center','FontSize',fs-3)

val = max(delta.rho_dt_norm);
leg_txt = [num2str(1000*max(delta.rho_dt),'%.1f'),'\times','10^{-3}'];
cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val))); % Index from the colormap
col2 = mean(cmap(cmap_ind,:),1); % Corresponding color
h(2) = bar(1.5,val,'FaceColor',col2,'EdgeColor',[0 0 0]);
text(1.5,1.4*val,leg_txt,'HorizontalAlignment','center','FontSize',fs-3)

val = min(delta.rho_dt_norm);
leg_txt = [num2str(1000*min(delta.rho_dt),'%.1f'),'\times','10^{-3}'];
cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val))); % Index from the colormap
col2 = mean(cmap(cmap_ind,:),1); % Corresponding color
h(2) = bar(1.5,val,'FaceColor',col2,'EdgeColor',[0 0 0]);
text(1.5,1.15*val,leg_txt,'HorizontalAlignment','center','FontSize',fs-3)

val = max(delta.SLR_norm);
leg_txt = num2str(max(delta.SLR),'%.1f');
cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val))); % Index from the colormap
col3 = mean(cmap(cmap_ind,:),1); % Corresponding color
h(3) = bar(2.5,val,'FaceColor',col3,'EdgeColor',[0 0 0]);
text(2.5,1.2*val,leg_txt,'HorizontalAlignment','center','FontSize',fs-3)

val = min(delta.SLR_norm);
leg_txt = num2str(min(delta.SLR),'%.1f');
cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val))); % Index from the colormap
col3 = mean(cmap(cmap_ind,:),1); % Corresponding color
h(3) = bar(2.5,val,'FaceColor',col3,'EdgeColor',[0 0 0]);
text(2.5,1.2*val,leg_txt,'HorizontalAlignment','center','FontSize',fs-3)

%     title('Legend')

axis([0 3 1.05*barmin 1.05*barmax])
set(gca,'XTickLabels',[],'YTickLabels',[],'FontSize',fs-3)

set(gcf,'Units','Inches');
pos = get(gcf,'Position');
set(gcf,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(gcf,['output/Threats_combined_bars_balance.pdf'],'-dpdf','-r500')
print(gcf,['output/Threats_combined_bars_balance.png'],'-dpng','-r500')

% OK.

%% Show bar graphs of combined threats, for all selected deltas (cmap = BALANCE), ROTATED

% Determine the y-axis range
barmin = min([min(delta.sed_norm),min(delta.rho_dt_norm),min(delta.SLR_norm)]);
barmax = max([max(delta.sed_norm),max(delta.rho_dt_norm),max(delta.SLR_norm)]);

% Choose a color scheme and prepare to color the bars according to value
cmap = cmocean('balance'); % (blue <0, white 0, red >0)
cmap_amp = max([abs(barmin),abs(barmax)]); % Maximal absolute bar value
cmap_val = linspace(-cmap_amp,cmap_amp,length(cmap)); % Bar values that each of the 256 elements in the cmap corresponds to

fs = 12; % FontSize

close(figure(17)); figure(17)
set(gcf,'units','normalized','outerposition',[0 0 1 1])

for i = 1:length(delta.lat) % Loop over all selected deltas
   
    subplot(6,8,i)
    
    % For each of the 3 factors (sediment retention, flood correlation trend, RSL), determine the
    % color that should be used to fill the respective bar

    % Sediment retention
    val = delta.sed_norm(i); % Value of this bar
    cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val))); % Find the index from the colormap that corresponds to this value
    col1 = mean(cmap(cmap_ind,:),1); % Corresponding color (average in the "1"-direction, in case there are more than one cmap_ind's

    % Idem for flood correlation trend
    val = delta.rho_dt_norm(i);
    cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val)));
    col2 = mean(cmap(cmap_ind,:),1);

    % Idem for RSL
    val = delta.SLR_norm(i);
    cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val)));
    col3 = mean(cmap(cmap_ind,:),1);

    % Plot the bars

        h(1) = bar(0.5,delta.sed_norm(i),      'FaceColor',col1,  'EdgeColor',[0 0 0]); hold on

    if delta.pval_dt(i) < 0.05 % If p<0.05, fill the flood trend bar fully with colors

        h(2) = bar(1.5,delta.rho_dt_norm(i),   'FaceColor',col2,  'EdgeColor',[0 0 0]);

    else % For p-value >= 0.05: distinguish between 0.05<=p<0.10 and p>=0.10

        if (delta.pval_dt(i) >= 0.05) && (delta.pval_dt(i) < 0.10) % For "weakly non-significant" trends: divide the bar into 6 almost touching bars
            nb = 6; % number of (almost touching) bars
        elseif (delta.pval_dt(i) >= 0.10) % For "strongly non-significant" trends: divide the bar into 4 bars with quite some space in between
            nb = 4; % number of separated bars
        else
            error('something wrong with p-values')
        end

        w = 0.1; % relative width of the individual bars
        bc = linspace(3*w/2, 1-3*w/2, nb); % individual bar center locations
        for nbi = 1:nb % Draw each individual bar
            bci = 1.0+bc(nbi); % Define the center of the currently drawn individual bar
            h(2) = bar(bci,delta.rho_dt_norm(i),w, 'FaceColor',col2,  'EdgeColor','none');
        end
        h(2) = bar(1.5,delta.rho_dt_norm(i),   'FaceColor','none','EdgeColor',[0 0 0]); % Surround the group of individual thin bars with a black edge

    end

    h(3) = bar(2.5,delta.SLR_norm(i),      'FaceColor',col3,  'EdgeColor',[0 0 0]);

    title(delta.plot_name(i))

    axis([0 3 1.05*barmin 1.05*barmax])
    set(gca,'XTickLabels',[],'YTickLabels',[],'FontSize',fs-3)

    if i == 23
        legend('Fluvial sediment retention [%]','Trend in surge-discharge correlation [yr^{-1}]','Relative sea-level change [mm yr^{-1}]','Position', [0.13 0.05 0.775 0.04],'NumColumns',3,'FontSize',fs-3,'FontWeight','bold')
    end

%     set(gca,'color','none')
    view([90 90])

end

% plot a legend

subplot(6,8,6*8)

val = max(delta.sed_norm);
leg_txt = num2str(100*max(delta.sed),'%.1f'); % Express in % in the legend
cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val))); % Index from the colormap
col1 = mean(cmap(cmap_ind,:),1); % Corresponding color
h(1) = bar(0.5,val,'FaceColor',col1,'EdgeColor',[0 0 0]); hold on
text(0.5,1.2*val,leg_txt,'HorizontalAlignment','center','FontSize',fs-3,'color',[0.2 0.2 0.2])

val = min(delta.sed_norm);
leg_txt = num2str(100*min(delta.sed),'%.1f'); % Express in % in the legend
cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val))); % Index from the colormap
col1 = mean(cmap(cmap_ind,:),1); % Corresponding color
h(1) = bar(0.5,val,'FaceColor',col1,'EdgeColor',[0 0 0]);
text(0.5,1.2*val,leg_txt,'HorizontalAlignment','center','FontSize',fs-3,'color',[0.2 0.2 0.2])

val = max(delta.rho_dt_norm);
leg_txt = [num2str(1000*max(delta.rho_dt),'%.1f'),'\times','10^{-3}'];
cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val))); % Index from the colormap
col2 = mean(cmap(cmap_ind,:),1); % Corresponding color
h(2) = bar(1.5,val,'FaceColor',col2,'EdgeColor',[0 0 0]);
text(1.5,1.2*val,leg_txt,'HorizontalAlignment','center','FontSize',fs-3,'color',[0.2 0.2 0.2])

val = min(delta.rho_dt_norm);
leg_txt = [num2str(1000*min(delta.rho_dt),'%.1f'),'\times','10^{-3}'];
cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val))); % Index from the colormap
col2 = mean(cmap(cmap_ind,:),1); % Corresponding color
h(2) = bar(1.5,val,'FaceColor',col2,'EdgeColor',[0 0 0]);
text(1.5,1.2*val,leg_txt,'HorizontalAlignment','center','FontSize',fs-3,'color',[0.2 0.2 0.2])

val = max(delta.SLR_norm);
leg_txt = num2str(max(delta.SLR),'%.1f');
cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val))); % Index from the colormap
col3 = mean(cmap(cmap_ind,:),1); % Corresponding color
h(3) = bar(2.5,val,'FaceColor',col3,'EdgeColor',[0 0 0]);
text(2.5,1.2*val,leg_txt,'HorizontalAlignment','center','FontSize',fs-3,'color',[0.2 0.2 0.2])

val = min(delta.SLR_norm);
leg_txt = num2str(min(delta.SLR),'%.1f');
cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val))); % Index from the colormap
col3 = mean(cmap(cmap_ind,:),1); % Corresponding color
h(3) = bar(2.5,val,'FaceColor',col3,'EdgeColor',[0 0 0]);
text(2.5,1.2*val,leg_txt,'HorizontalAlignment','center','FontSize',fs-3,'color',[0.2 0.2 0.2])

%     title('Legend')

axis([0 3 1.05*barmin 1.05*barmax])
set(gca,'XTickLabels',[],'YTickLabels',[],'FontSize',fs-3)
view([90 90])

set(gcf,'Units','Inches');
% set(gcf,'color','none')
% set(gca,'color','none')

pos = get(gcf,'Position');
set(gcf,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(gcf,['output/Threats_combined_bars_balance_rotated.pdf'],'-dpdf','-r500')
print(gcf,['output/Threats_combined_bars_balance_rotated.png'],'-dpng','-r500')

% OK.

%% Embed all bar diagrams in a worldmap
    
close(figure(18));figure(18)

% Set up plot window size such that the figure itself is 17.8cm wide (= text width)
window_width = 26.74;
window_height = 16.49;

set(gcf,'units','centimeters','outerposition',[2 2 window_width window_height],'color','w')

fs = 8; % FontSize (in points)

% Make a world map with the locations of all selected deltas indicated

delta.lon(delta.lon>180) = delta.lon(delta.lon>180) - 360; % Now make sure all longitudes are between -180 and 180 degrees again

ax = worldmap(latlim,lonlim);
setm(ax,'mapprojection','eqdcylin');
land = shaperead('landareas.shp', 'UseGeoCoords', true);
geoshow(ax, land, 'FaceColor', [0.9 0.9 0.9],'EdgeColor','none')
hold on
scatterm(delta.lat,delta.lon,10,[0.5 0.5 0.5],'Filled')
    
plabel off
mlabel off
gridm off

% Plot connector lines
for i = 1:length(delta.lat)
    if delta.plot_name(i) == 'Yukon'
        con_length = 7; % Connector length, in degrees
        con_deg = 230; % Angle that connector should make
    elseif delta.plot_name(i) == 'MacKenzie'
        con_length = 7; % Connector length, in degrees
        con_deg = 170; % Angle that connector should make
    elseif delta.plot_name(i) == 'Colorado'
        con_length = 20; % Connector length, in degrees
        con_deg = 150; % Angle that connector should make
    elseif delta.plot_name(i) == 'Mississippi'
        con_length = 15; % Connector length, in degrees
        con_deg = 130; % Angle that connector should make
    elseif delta.plot_name(i) == 'Rio Grande'
        con_length = 25; % Connector length, in degrees
        con_deg = 200; % Angle that connector should make
    elseif delta.plot_name(i) == 'Grijalva'
        con_length = 30; % Connector length, in degrees
        con_deg = 230; % Angle that connector should make
    elseif delta.plot_name(i) == 'Magdalena'
        con_length = 7; % Connector length, in degrees
        con_deg = 70; % Angle that connector should make
    elseif delta.plot_name(i) == 'Orinoco'
        con_length = 28; % Connector length, in degrees
        con_deg = 200; % Angle that connector should make
    elseif delta.plot_name(i) == 'Amazon'
        con_length = 20; % Connector length, in degrees
        con_deg = 210; % Angle that connector should make
    elseif delta.plot_name(i) == 'Sao Francisco'
        con_length = 15; % Connector length, in degrees
        con_deg = 240; % Angle that connector should make
    elseif delta.plot_name(i) == 'Parana'
        con_length = 7; % Connector length, in degrees
        con_deg = 210; % Angle that connector should make
    elseif delta.plot_name(i) == 'Vistula'
        con_length = 10; % Connector length, in degrees
        con_deg = 40; % Angle that connector should make
    elseif delta.plot_name(i) == 'Rhine-Meuse'
        con_length = 15; % Connector length, in degrees
        con_deg = 90; % Angle that connector should make
    elseif delta.plot_name(i) == 'Rhone'
        con_length = 20; % Connector length, in degrees
        con_deg = 135; % Angle that connector should make
    elseif delta.plot_name(i) == 'Po'
        con_length = 10; % Connector length, in degrees
        con_deg = -35; % Angle that connector should make
    elseif delta.plot_name(i) == 'Ebro'
        con_length = 20; % Connector length, in degrees
        con_deg = 170; % Angle that connector should make
    elseif delta.plot_name(i) == 'Moulouya'
        con_length = 15; % Connector length, in degrees
        con_deg = 275; % Angle that connector should make
    elseif delta.plot_name(i) == 'Sebou'
        con_length = 25; % Connector length, in degrees
        con_deg = 180; % Angle that connector should make
    elseif delta.plot_name(i) == 'Senegal'
        con_length = 15; % Connector length, in degrees
        con_deg = 185; % Angle that connector should make
    elseif delta.plot_name(i) == 'Volta'
        con_length = 15; % Connector length, in degrees
        con_deg = 200; % Angle that connector should make
    elseif delta.plot_name(i) == 'Niger'
        con_length = 10; % Connector length, in degrees
        con_deg = 45; % Angle that connector should make
    elseif delta.plot_name(i) == 'Congo'
        con_length = 15; % Connector length, in degrees
        con_deg = 200; % Angle that connector should make
    elseif delta.plot_name(i) == 'Limpopo'
        con_length = 15; % Connector length, in degrees
        con_deg = 190; % Angle that connector should make
    elseif delta.plot_name(i) == 'Zambezi'
        con_length = 15; % Connector length, in degrees
        con_deg = 330; % Angle that connector should make
    elseif delta.plot_name(i) == 'Tana'
        con_length = 10; % Connector length, in degrees
        con_deg = 200; % Angle that connector should make
    elseif delta.plot_name(i) == 'Nile'
        con_length = 15; % Connector length, in degrees
        con_deg = -85; % Angle that connector should make
    elseif delta.plot_name(i) == 'Tigris'
        con_length = 15; % Connector length, in degrees
        con_deg = 135; % Angle that connector should make
    elseif delta.plot_name(i) == 'Indus'
        con_length = 20; % Connector length, in degrees
        con_deg = 210; % Angle that connector should make
    elseif delta.plot_name(i) == 'Krishna'
        con_length = 15; % Connector length, in degrees
        con_deg = 220; % Angle that connector should make
    elseif delta.plot_name(i) == 'Godavari'
        con_length = 30; % Connector length, in degrees
        con_deg = 275; % Angle that connector should make
    elseif delta.plot_name(i) == 'Mahanadi'
        con_length = 30; % Connector length, in degrees
        con_deg = 155; % Angle that connector should make
    elseif delta.plot_name(i) == 'Ganges-Brahmaputra'
        con_length = 30; % Connector length, in degrees
        con_deg = 125; % Angle that connector should make
    elseif delta.plot_name(i) == 'Chao Phraya'
        con_length = 25; % Connector length, in degrees
        con_deg = 110; % Angle that connector should make
    elseif delta.plot_name(i) == 'Irriwaddy'
        con_length = 20; % Connector length, in degrees
        con_deg = -85; % Angle that connector should make
    elseif delta.plot_name(i) == 'Mekong'
        con_length = 15; % Connector length, in degrees
        con_deg = -20; % Angle that connector should make
    elseif delta.plot_name(i) == 'Red'
        con_length = 10; % Connector length, in degrees
        con_deg = 90; % Angle that connector should make
    elseif delta.plot_name(i) == 'Mahakam'
        con_length = 20; % Connector length, in degrees
        con_deg = -80; % Angle that connector should make
    elseif delta.plot_name(i) == 'Pearl'
        con_length = 25; % Connector length, in degrees
        con_deg = -15; % Angle that connector should make
    elseif delta.plot_name(i) == 'Yangtze'
        con_length = 15; % Connector length, in degrees
        con_deg = 90; % Angle that connector should make
    elseif delta.plot_name(i) == 'Yellow'
        con_length = 25; % Connector length, in degrees
        con_deg = 130; % Angle that connector should make
    elseif delta.plot_name(i) == 'Han'
        con_length = 8; % Connector length, in degrees
        con_deg = -5; % Angle that connector should make
    elseif delta.plot_name(i) == 'Tone'
        con_length = 15; % Connector length, in degrees
        con_deg = -10; % Angle that connector should make
    elseif delta.plot_name(i) == 'Amur'
        con_length = 20; % Connector length, in degrees
        con_deg = 5; % Angle that connector should make
    elseif delta.plot_name(i) == 'Lena'
        con_length = 15; % Connector length, in degrees
        con_deg = -10; % Angle that connector should make
    elseif delta.plot_name(i) == 'Fly'
        con_length = 15; % Connector length, in degrees
        con_deg = 25; % Angle that connector should make
    elseif delta.plot_name(i) == 'Burdekin'
        con_length = 15; % Connector length, in degrees
        con_deg = -10; % Angle that connector should make
    elseif delta.plot_name(i) == 'Murray'
        con_length = 15; % Connector length, in degrees
        con_deg = -45; % Angle that connector should make
    end
    % Plot the actual connectors
    delta.conlat(i) = delta.lat(i) + sind(con_deg)*con_length;
    delta.conlon(i) = delta.lon(i) + cosd(con_deg)*con_length;
    geoshow([delta.lat(i),delta.conlat(i)]',[delta.lon(i),delta.conlon(i)]','Color',[0.5 0.5 0.5])
end

%%%%%%%%%%%% Show the bar plots inside the world map %%%%%%%%%%%%%%%%%%%%%%

% Determine the y-axis range
barmin = min([min(delta.sed_norm),min(delta.rho_dt_norm),min(delta.SLR_norm)]);
barmax = max([max(delta.sed_norm),max(delta.rho_dt_norm),max(delta.SLR_norm)]);

% Choose a color scheme and prepare to color the bars according to value
cmap = cmocean('balance'); % (blue <0, white 0, red >0)
cmap_amp = max([abs(barmin),abs(barmax)]); % Maximal absolute bar value
cmap_val = linspace(-cmap_amp,cmap_amp,length(cmap)); % Bar values that each of the 256 elements in the cmap corresponds to

% Manually determine the relative position of the world map compared to the entire plot window
left    = 3.58/19.82;
bottom  = 2.53/10.64;
width   = 13.33/19.82;
height  = 6.23/10.64;

% Set the relative width/height of each bar plot
barwidth = 0.06;
barheight = barwidth;

% Plot the bargraphs in the worldmap

for i = 1:length(delta.lat) % Loop over all selected deltas
       
    % Determine the relative x and y position of each river delta (that is, convert the (lat,lon) pairs to relative (x,y) coordinates) 
    x_rel = left   + width*  ( (delta.conlon(i) - lonlim(1))/(lonlim(2) - lonlim(1)) );
    y_rel = bottom + height* ( (delta.conlat(i) - latlim(1))/(latlim(2) - latlim(1)) );

    barax(i) = axes('Position',[x_rel-0.5*barwidth y_rel-0.5*barheight barwidth barheight]);

    % For each of the 3 factors (sediment retention, flood correlation trend, RSL), determine the
    % color that should be used to fill the respective bar

    % Sediment retention
    val = delta.sed_norm(i); % Value of this bar
    cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val))); % Find the index from the colormap that corresponds to this value
    col1 = mean(cmap(cmap_ind,:),1); % Corresponding color (average in the "1"-direction, in case there are more than one cmap_ind's

    % Idem for flood correlation trend
    val = delta.rho_dt_norm(i);
    cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val)));
    col2 = mean(cmap(cmap_ind,:),1);

    % Idem for RSL
    val = delta.SLR_norm(i);
    cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val)));
    col3 = mean(cmap(cmap_ind,:),1);

    % Plot the bars

    h(1) = bar(0.5,delta.sed_norm(i),      'FaceColor',col1,  'EdgeColor',[0 0 0]); hold on

    if delta.pval_dt(i) < 0.05 % If p<0.05, fill the flood trend bar fully with colors

        h(2) = bar(1.5,delta.rho_dt_norm(i),   'FaceColor',col2,  'EdgeColor',[0 0 0]);

    else % For p-value >= 0.05: distinguish between 0.05<=p<0.10 and p>=0.10

        if (delta.pval_dt(i) >= 0.05) && (delta.pval_dt(i) < 0.10) % For "weakly non-significant" trends: divide the bar into 6 almost touching bars
            nb = 6; % number of (almost touching) bars
        elseif (delta.pval_dt(i) >= 0.10) % For "strongly non-significant" trends: divide the bar into 4 bars with quite some space in between
            nb = 4; % number of separated bars
        else
            error('something wrong with p-values')
        end

        h(2) = bar(1.5,delta.rho_dt_norm(i),   'FaceColor','w','EdgeColor','none'); % Put a white big bar underneath
        w = 0.1; % relative width of the individual bars
        bc = linspace(3*w/2, 1-3*w/2, nb); % individual bar center locations
        for nbi = 1:nb % Draw each individual bar
            bci = 1.0+bc(nbi); % Define the center of the currently drawn individual bar
            h(2) = bar(bci,delta.rho_dt_norm(i),w, 'FaceColor',col2,  'EdgeColor','none');
        end
        h(2) = bar(1.5,delta.rho_dt_norm(i),   'FaceColor','none','EdgeColor',[0 0 0]); % Surround the group of individual thin bars with a black edge

    end

    h(3) = bar(2.5,delta.SLR_norm(i),      'FaceColor',col3,  'EdgeColor',[0 0 0]);

    title(delta.plot_name(i))%,'FontSize',fs)

    axis([-0.1 3 1.0*barmin 1.0*barmax])
    set(gca,'XTickLabels',[],'YTickLabels',[])%,'FontSize',fs)

    set(gca,'color','none','box','off','XColor','none','YAxisLocation','right','YTick',[])
    view([90 90])
    pbaspect([1 1 1])

end

% Legend

x_rel = left + 0.4*width;
y_rel = 1.04*bottom;

% Position for text strings in legend
lr = 2.7;
ll = -1.5;
lt = 3.2;

% Create a white rectangle as background
axes('Position',[0.965*x_rel 0.99*y_rel 5.3*barwidth 1.5*barheight],'color','w','XTick',[],'YTick',[],'XColor','none','YColor','none')

% Create the actual legend
axes('Position',[x_rel 1.020*y_rel barwidth barheight])

val = max(delta.sed_norm);
leg_txt = num2str(100*max(delta.sed),'%.1f'); % Express in % in the legend
cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val))); % Index from the colormap
col1 = mean(cmap(cmap_ind,:),1); % Corresponding color
h(1) = bar(0.5,val,'FaceColor',col1,'EdgeColor',[0 0 0]); hold on
text(0.5,lr,leg_txt,'HorizontalAlignment','right','VerticalAlignment','middle')%,'FontSize',fs)

val = min(delta.sed_norm);
leg_txt = num2str(100*min(delta.sed),'%.1f'); % Express in % in the legend
cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val))); % Index from the colormap
col1 = mean(cmap(cmap_ind,:),1); % Corresponding color
h(1) = bar(0.5,val,'FaceColor',col1,'EdgeColor',[0 0 0]);
text(0.5,ll,leg_txt,'HorizontalAlignment','right','VerticalAlignment','middle')%,'FontSize',fs)

val = max(delta.rho_dt_norm);
leg_txt = [num2str(1000*max(delta.rho_dt),'%.1f')];
cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val))); % Index from the colormap
col2 = mean(cmap(cmap_ind,:),1); % Corresponding color
h(2) = bar(1.5,val,'FaceColor',col2,'EdgeColor',[0 0 0]);
text(1.5,lr,leg_txt,'HorizontalAlignment','right','VerticalAlignment','middle')%,'FontSize',fs)

val = min(delta.rho_dt_norm);
leg_txt = [num2str(1000*min(delta.rho_dt),'%.1f')];
cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val))); % Index from the colormap
col2 = mean(cmap(cmap_ind,:),1); % Corresponding color
h(2) = bar(1.5,val,'FaceColor',col2,'EdgeColor',[0 0 0]);
text(1.5,ll,leg_txt,'HorizontalAlignment','right','VerticalAlignment','middle')%,'FontSize',fs)

val = max(delta.SLR_norm);
leg_txt = num2str(max(delta.SLR),'%.1f');
cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val))); % Index from the colormap
col3 = mean(cmap(cmap_ind,:),1); % Corresponding color
h(3) = bar(2.5,val,'FaceColor',col3,'EdgeColor',[0 0 0]);
text(2.5,lr,leg_txt,'HorizontalAlignment','right','VerticalAlignment','middle')%,'FontSize',fs)

val = min(delta.SLR_norm);
leg_txt = [num2str(min(delta.SLR),'%.1f')];
cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val))); % Index from the colormap
col3 = mean(cmap(cmap_ind,:),1); % Corresponding color
h(3) = bar(2.5,val,'FaceColor',col3,'EdgeColor',[0 0 0]);
text(2.5,ll,leg_txt,'HorizontalAlignment','right','VerticalAlignment','middle')%,'FontSize',fs)

leg_sed = 'Fluvial sediment retention [%]';
text(0.5,lt,leg_sed,'HorizontalAlignment','left','VerticalAlignment','middle')%,'FontSize',fs)

leg_flood = ['Trend in surge-discharge correlation [10^{-3} yr^{-1}]'];
text(1.5,lt,leg_flood,'HorizontalAlignment','left','VerticalAlignment','middle')%,'FontSize',fs)

leg_rsl = 'Relative sea-level change [mm yr^{-1}]';
text(2.5,lt,leg_rsl,'HorizontalAlignment','left','VerticalAlignment','middle')%,'FontSize',fs)

title('Legend')%,'FontSize',fs)
axis([-0.1 3 1.0*barmin 1.0*barmax])
set(gca,'XTickLabels',[],'YTickLabels',[])%,'FontSize',fs)

set(gca,'color','none','box','off','XColor','none','YAxisLocation','right','YTick',[])
view([90 90])%% Embed all bar diagrams in a worldmap
pbaspect([1 1 1])

%%%% Create a second legend, to indicate the significance levels in rho_dt
x_rel = left + 0.01*width;
y_rel = 1.04*bottom;

% Position for text strings in legend
lr = 2.7;
ll = -1.5;
lt = 1.2;

% Create a white rectangle as background
axes('Position',[0.995*x_rel 0.99*y_rel 2.2*barwidth 1.5*barheight],'color','w','XTick',[],'YTick',[],'XColor','none','YColor','none')

% Create the actual legend
axes('Position',[x_rel 1.020*y_rel barwidth barheight])

% p<0.05 %%%%%%%%%%%%%%%%%%

val = 0.5; %max(delta.sed_norm);
cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val))); % Index from the colormap
col1 = mean(cmap(cmap_ind,:),1); % Corresponding color
h(1) = bar(0.5,val,'FaceColor',col1,'EdgeColor',[0 0 0]); hold on

val = -0.5; %min(delta.sed_norm);
cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val))); % Index from the colormap
col1 = mean(cmap(cmap_ind,:),1); % Corresponding color
h(1) = bar(0.5,val,'FaceColor',col1,'EdgeColor',[0 0 0]);

% 0.10<=p<0.05 %%%%%%%%%%%%%%%%%%

val = 0.5; %max(delta.rho_dt_norm);
cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val))); % Index from the colormap
col2 = mean(cmap(cmap_ind,:),1); % Corresponding color
nb = 6;
h(2) = bar(1.5,val,'FaceColor','w','EdgeColor','none'); % Put a white big bar underneath
w = 0.1; % relative width of the individual bars
bc = linspace(3*w/2, 1-3*w/2, nb); % individual bar center locations
for nbi = 1:nb % Draw each individual bar
    bci = 1.0+bc(nbi); % Define the center of the currently drawn individual bar
    h(2) = bar(bci,val,w, 'FaceColor',col2,  'EdgeColor','none');
end
h(2) = bar(1.5,val,   'FaceColor','none','EdgeColor',[0 0 0]); % Surround the group of individual thin bars with a black edge

val = -0.5; %min(delta.rho_dt_norm);
cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val))); % Index from the colormap
col2 = mean(cmap(cmap_ind,:),1); % Corresponding color
nb = 6;
h(2) = bar(1.5,val,'FaceColor','w','EdgeColor','none'); % Put a white big bar underneath
w = 0.1; % relative width of the individual bars
bc = linspace(3*w/2, 1-3*w/2, nb); % individual bar center locations
for nbi = 1:nb % Draw each individual bar
    bci = 1.0+bc(nbi); % Define the center of the currently drawn individual bar
    h(2) = bar(bci,val,w, 'FaceColor',col2,  'EdgeColor','none');
end
h(2) = bar(1.5,val,   'FaceColor','none','EdgeColor',[0 0 0]); % Surround the group of individual thin bars with a black edge

% p > 0.10 %%%%%%%%%%%%%%%%%%

val = 0.5; %max(delta.SLR_norm);
cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val))); % Index from the colormap
col3 = mean(cmap(cmap_ind,:),1); % Corresponding color
nb = 4;
h(3) = bar(2.5,val,'FaceColor','w','EdgeColor','none'); % Put a white big bar underneath
w = 0.1; % relative width of the individual bars
bc = linspace(3*w/2, 1-3*w/2, nb); % individual bar center locations
for nbi = 1:nb % Draw each individual bar
    bci = 2.0+bc(nbi); % Define the center of the currently drawn individual bar
    h(3) = bar(bci,val,w, 'FaceColor',col3,  'EdgeColor','none');
end
h(3) = bar(2.5,val,   'FaceColor','none','EdgeColor',[0 0 0]); % Surround the group of individual thin bars with a black edge

val = -0.5; %min(delta.SLR_norm);
cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val))); % Index from the colormap
col3 = mean(cmap(cmap_ind,:),1); % Corresponding color
nb = 4;
h(3) = bar(2.5,val,'FaceColor','w','EdgeColor','none'); % Put a white big bar underneath
w = 0.1; % relative width of the individual bars
bc = linspace(3*w/2, 1-3*w/2, nb); % individual bar center locations
for nbi = 1:nb % Draw each individual bar
    bci = 2.0+bc(nbi); % Define the center of the currently drawn individual bar
    h(3) = bar(bci,val,w, 'FaceColor',col3,  'EdgeColor','none');
end
h(3) = bar(2.5,val,   'FaceColor','none','EdgeColor',[0 0 0]); % Surround the group of individual thin bars with a black edge

% Add labels to this legend
leg_plow = ['p < 0.05'];
text(0.5,lt,leg_plow,'HorizontalAlignment','left','VerticalAlignment','middle')%,'FontSize',fs)

leg_pmid = ['0.05 \leq p < 0.10'];
text(1.5,lt,leg_pmid,'HorizontalAlignment','left','VerticalAlignment','middle')%,'FontSize',fs)

leg_phigh = ['p \geq 0.10'];
text(2.5,lt,leg_phigh,'HorizontalAlignment','left','VerticalAlignment','middle')%,'FontSize',fs)

title('                    Significance of trend')%,']Position',)%,'FontSize',fs)
axis([-0.1 3 1.0*barmin 1.0*barmax])
set(gca,'XTickLabels',[],'YTickLabels',[])%,'FontSize',fs)

set(gca,'color','none','box','off','XColor','none','YAxisLocation','right','YTick',[])
view([90 90])%% Embed all bar diagrams in a worldmap
pbaspect([1 1 1])

% Save plot
    
% fontsize(gcf,fs,"points")
% fontsize(gca,fs,"points")
% fontname(gcf,'Helvetica')
% fontname(gca,'Helvetica')
% set(gcf,'FontWeight', 'Normal')
set(findall(gcf, 'Type', 'Text'),'FontWeight', 'Normal','FontName','Helvetica','FontSize',fs,'units','points')
set(findall(gca, 'Type', 'Text'),'FontWeight', 'Normal','FontName','Helvetica','FontSize',fs,'units','points')

savename = 'Threats_combined_bars_balance_rotated_worldmap';
dpi = 300;
exportgraphics(gcf,['output/',savename,'.pdf'],'Resolution',dpi,'ContentType','vector')
exportgraphics(gcf,['output/',savename,'.png'],'Resolution',dpi,'ContentType','vector')

% OK.

%% Clear all obsolete values

clearvars -except CorrType Couasnon2020NHESS delta latlim lonlim Nienhuis2023AREPS wgs84

save(['output','/','ReviewPaper_Worldmap_v1_output.mat'])

% OK.

% END OF THE SCRIPT. %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% If you don't want to rerun the entire script: directly load the previously computed output
%  
% addpath cmocean
% addpath output
% 
% load(['output','/','ReviewPaper_Worldmap_v1_output.mat'])

% OK.

%% Load color schemes from Nature Reviews

col.red1 = [248,208,204]/255;
col.red2 = [238,164,166]/255;
col.red3 = [223,103,101]/255;
col.red4 = [202, 62, 61]/255;
col.red5 = [155, 38, 37]/255;
col.red6 = [115, 29, 23]/255;

col.blu1 = [201,230,247]/255;
col.blu2 = [156,204,236]/255;
col.blu3 = [ 87,154,209]/255;
col.blu4 = [ 10,115,176]/255;
col.blu5 = [  7, 81,142]/255;
col.blu6 = [ 22, 48, 91]/255;

col.gry1 = [230,232,238]/255;
col.gry2 = [201,205,219]/255;
col.gry3 = [154,164,182]/255;
col.gry4 = [112,124,146]/255;
col.gry5 = [ 71, 85,107]/255;
col.gry6 = [ 35, 50, 71]/255;

% OK.

%% Embed all bar diagrams in a worldmap - optimized legend
    
close(figure(19));figure(19)

% Set up plot window size such that the figure itself is 17.8cm wide (= text width)
window_width = 26.74;
window_height = 16.49;

set(gcf,'units','centimeters','outerposition',[2 2 window_width window_height],'color','w')

fs = 8; % FontSize (in points)

% Make a world map with the locations of all selected deltas indicated

delta.lon(delta.lon>180) = delta.lon(delta.lon>180) - 360; % Now make sure all longitudes are between -180 and 180 degrees again

ax = worldmap(latlim,lonlim);
setm(ax,'mapprojection','eqdcylin');
land = shaperead('landareas.shp', 'UseGeoCoords', true);
geoshow(ax, land, 'FaceColor', [0.9 0.9 0.9],'EdgeColor','none')
hold on
scatterm(delta.lat,delta.lon,10,[0.5 0.5 0.5],'Filled')
    
plabel off
mlabel off
gridm off

% Plot connector lines
for i = 1:length(delta.lat)
    if delta.plot_name(i) == 'Yukon'
        con_length = 7; % Connector length, in degrees
        con_deg = 230; % Angle that connector should make
    elseif delta.plot_name(i) == 'MacKenzie'
        con_length = 7; % Connector length, in degrees
        con_deg = 170; % Angle that connector should make
    elseif delta.plot_name(i) == 'Colorado'
        con_length = 20; % Connector length, in degrees
        con_deg = 150; % Angle that connector should make
    elseif delta.plot_name(i) == 'Mississippi'
        con_length = 15; % Connector length, in degrees
        con_deg = 130; % Angle that connector should make
    elseif delta.plot_name(i) == 'Rio Grande'
        con_length = 25; % Connector length, in degrees
        con_deg = 200; % Angle that connector should make
    elseif delta.plot_name(i) == 'Grijalva'
        con_length = 30; % Connector length, in degrees
        con_deg = 230; % Angle that connector should make
    elseif delta.plot_name(i) == 'Magdalena'
        con_length = 7; % Connector length, in degrees
        con_deg = 70; % Angle that connector should make
    elseif delta.plot_name(i) == 'Orinoco'
        con_length = 28; % Connector length, in degrees
        con_deg = 200; % Angle that connector should make
    elseif delta.plot_name(i) == 'Amazon'
        con_length = 20; % Connector length, in degrees
        con_deg = 210; % Angle that connector should make
    elseif delta.plot_name(i) == 'Sao Francisco'
        con_length = 15; % Connector length, in degrees
        con_deg = 240; % Angle that connector should make
    elseif delta.plot_name(i) == 'Parana'
        con_length = 7; % Connector length, in degrees
        con_deg = 210; % Angle that connector should make
    elseif delta.plot_name(i) == 'Vistula'
        con_length = 10; % Connector length, in degrees
        con_deg = 40; % Angle that connector should make
    elseif delta.plot_name(i) == 'Rhine-Meuse'
        con_length = 15; % Connector length, in degrees
        con_deg = 90; % Angle that connector should make
    elseif delta.plot_name(i) == 'Rhone'
        con_length = 20; % Connector length, in degrees
        con_deg = 135; % Angle that connector should make
    elseif delta.plot_name(i) == 'Po'
        con_length = 10; % Connector length, in degrees
        con_deg = -35; % Angle that connector should make
    elseif delta.plot_name(i) == 'Ebro'
        con_length = 20; % Connector length, in degrees
        con_deg = 170; % Angle that connector should make
    elseif delta.plot_name(i) == 'Moulouya'
        con_length = 15; % Connector length, in degrees
        con_deg = 275; % Angle that connector should make
    elseif delta.plot_name(i) == 'Sebou'
        con_length = 25; % Connector length, in degrees
        con_deg = 180; % Angle that connector should make
    elseif delta.plot_name(i) == 'Senegal'
        con_length = 15; % Connector length, in degrees
        con_deg = 185; % Angle that connector should make
    elseif delta.plot_name(i) == 'Volta'
        con_length = 15; % Connector length, in degrees
        con_deg = 200; % Angle that connector should make
    elseif delta.plot_name(i) == 'Niger'
        con_length = 10; % Connector length, in degrees
        con_deg = 45; % Angle that connector should make
    elseif delta.plot_name(i) == 'Congo'
        con_length = 15; % Connector length, in degrees
        con_deg = 200; % Angle that connector should make
    elseif delta.plot_name(i) == 'Limpopo'
        con_length = 15; % Connector length, in degrees
        con_deg = 190; % Angle that connector should make
    elseif delta.plot_name(i) == 'Zambezi'
        con_length = 15; % Connector length, in degrees
        con_deg = 330; % Angle that connector should make
    elseif delta.plot_name(i) == 'Tana'
        con_length = 10; % Connector length, in degrees
        con_deg = 200; % Angle that connector should make
    elseif delta.plot_name(i) == 'Nile'
        con_length = 15; % Connector length, in degrees
        con_deg = -85; % Angle that connector should make
    elseif delta.plot_name(i) == 'Tigris'
        con_length = 15; % Connector length, in degrees
        con_deg = 135; % Angle that connector should make
    elseif delta.plot_name(i) == 'Indus'
        con_length = 20; % Connector length, in degrees
        con_deg = 210; % Angle that connector should make
    elseif delta.plot_name(i) == 'Krishna'
        con_length = 15; % Connector length, in degrees
        con_deg = 220; % Angle that connector should make
    elseif delta.plot_name(i) == 'Godavari'
        con_length = 30; % Connector length, in degrees
        con_deg = 275; % Angle that connector should make
    elseif delta.plot_name(i) == 'Mahanadi'
        con_length = 30; % Connector length, in degrees
        con_deg = 155; % Angle that connector should make
    elseif delta.plot_name(i) == 'Ganges-Brahmaputra'
        con_length = 30; % Connector length, in degrees
        con_deg = 125; % Angle that connector should make
    elseif delta.plot_name(i) == 'Chao Phraya'
        con_length = 25; % Connector length, in degrees
        con_deg = 110; % Angle that connector should make
    elseif delta.plot_name(i) == 'Irriwaddy'
        con_length = 20; % Connector length, in degrees
        con_deg = -85; % Angle that connector should make
    elseif delta.plot_name(i) == 'Mekong'
        con_length = 15; % Connector length, in degrees
        con_deg = -20; % Angle that connector should make
    elseif delta.plot_name(i) == 'Red'
        con_length = 10; % Connector length, in degrees
        con_deg = 90; % Angle that connector should make
    elseif delta.plot_name(i) == 'Mahakam'
        con_length = 20; % Connector length, in degrees
        con_deg = -80; % Angle that connector should make
    elseif delta.plot_name(i) == 'Pearl'
        con_length = 25; % Connector length, in degrees
        con_deg = -15; % Angle that connector should make
    elseif delta.plot_name(i) == 'Yangtze'
        con_length = 15; % Connector length, in degrees
        con_deg = 90; % Angle that connector should make
    elseif delta.plot_name(i) == 'Yellow'
        con_length = 25; % Connector length, in degrees
        con_deg = 130; % Angle that connector should make
    elseif delta.plot_name(i) == 'Han'
        con_length = 8; % Connector length, in degrees
        con_deg = -5; % Angle that connector should make
    elseif delta.plot_name(i) == 'Tone'
        con_length = 15; % Connector length, in degrees
        con_deg = -10; % Angle that connector should make
    elseif delta.plot_name(i) == 'Amur'
        con_length = 20; % Connector length, in degrees
        con_deg = 5; % Angle that connector should make
    elseif delta.plot_name(i) == 'Lena'
        con_length = 15; % Connector length, in degrees
        con_deg = -10; % Angle that connector should make
    elseif delta.plot_name(i) == 'Fly'
        con_length = 15; % Connector length, in degrees
        con_deg = 25; % Angle that connector should make
    elseif delta.plot_name(i) == 'Burdekin'
        con_length = 15; % Connector length, in degrees
        con_deg = -10; % Angle that connector should make
    elseif delta.plot_name(i) == 'Murray'
        con_length = 15; % Connector length, in degrees
        con_deg = -45; % Angle that connector should make
    end
    % Plot the actual connectors
    delta.conlat(i) = delta.lat(i) + sind(con_deg)*con_length;
    delta.conlon(i) = delta.lon(i) + cosd(con_deg)*con_length;
    geoshow([delta.lat(i),delta.conlat(i)]',[delta.lon(i),delta.conlon(i)]','Color',[0.5 0.5 0.5])
end

%%%%%%%%%%%% Show the bar plots inside the world map %%%%%%%%%%%%%%%%%%%%%%

% Determine the y-axis range
barmin = min([min(delta.sed_norm_abs),min(delta.rho_dt_norm_abs),min(delta.SLR_norm_abs)]);
barmax = max([max(delta.sed_norm_abs),max(delta.rho_dt_norm_abs),max(delta.SLR_norm_abs)]);

% Choose a color scheme and prepare to color the bars according to value
cmap = cmocean('balance'); % (blue <0, white 0, red >0)
cmap_amp = max([abs(barmin),abs(barmax)]); % Maximal absolute bar value
cmap_val = linspace(-cmap_amp,cmap_amp,length(cmap)); % Bar values that each of the 256 elements in the cmap corresponds to

% Set fixed colors for each of the 3 bars, and distinguish <0 (blue) from >0 (red)

val = 0.7; % Value of this bar 
cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val))); % Find the index from the colormap that corresponds to this value
col1_pls = mean(cmap(cmap_ind,:),1); % Corresponding color (average in the "1"-direction, in case there are more than one cmap_ind's

val = -0.7; % Value of this bar
cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val))); % Find the index from the colormap that corresponds to this value
col1_min = mean(cmap(cmap_ind,:),1); % Corresponding color (average in the "1"-direction, in case there are more than one cmap_ind's

val = 0.5; % Value of this bar
cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val))); % Find the index from the colormap that corresponds to this value
col2_pls = mean(cmap(cmap_ind,:),1); % Corresponding color (average in the "1"-direction, in case there are more than one cmap_ind's

val = -0.5; % Value of this bar
cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val))); % Find the index from the colormap that corresponds to this value
col2_min = mean(cmap(cmap_ind,:),1); % Corresponding color (average in the "1"-direction, in case there are more than one cmap_ind's

val = 0.3; % Value of this bar
cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val))); % Find the index from the colormap that corresponds to this value
col3_pls = mean(cmap(cmap_ind,:),1); % Corresponding color (average in the "1"-direction, in case there are more than one cmap_ind's

val = -0.3; % Value of this bar
cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val))); % Find the index from the colormap that corresponds to this value
col3_min = mean(cmap(cmap_ind,:),1); % Corresponding color (average in the "1"-direction, in case there are more than one cmap_ind's

% Manually determine the relative position of the world map compared to the entire plot window
left    = 3.58/19.82;
bottom  = 2.53/10.64;
width   = 13.33/19.82;
height  = 6.23/10.64;

% Set the relative width/height of each bar plot
barwidth = 0.06;
barheight = barwidth;

% Plot the bargraphs in the worldmap

for i = 1:length(delta.lat) % Loop over all selected deltas
       
    % Determine the relative x and y position of each river delta (that is, convert the (lat,lon) pairs to relative (x,y) coordinates) 
    x_rel = left   + width*  ( (delta.conlon(i) - lonlim(1))/(lonlim(2) - lonlim(1)) );
    y_rel = bottom + height* ( (delta.conlat(i) - latlim(1))/(latlim(2) - latlim(1)) );

    barax(i) = axes('Position',[x_rel-0.5*barwidth y_rel-0.5*barheight barwidth barheight]);

    % For each of the 3 factors (sediment retention, flood correlation trend, RSL), determine the
    % color that should be used to fill the respective bar

    % Sediment retention
    val = delta.sed_norm_abs(i); % Value of this bar
%     cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val))); % Find the index from the colormap that corresponds to this value
%     col1 = mean(cmap(cmap_ind,:),1); % Corresponding color (average in the "1"-direction, in case there are more than one cmap_ind's
    if val < 0
        col1 = col1_min;
    else
        col1 = col1_pls;
    end

    % Idem for flood correlation trend
    val = delta.rho_dt_norm_abs(i);
%     cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val)));
%     col2 = mean(cmap(cmap_ind,:),1);
    if val < 0
        col2 = col2_min;
    else
        col2 = col2_pls;
    end

    % Idem for RSL
    val = delta.SLR_norm_abs(i);
%     cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val)));
%     col3 = mean(cmap(cmap_ind,:),1);
    if val < 0
        col3 = col3_min;
    else
        col3 = col3_pls;
    end

    % Plot the bars

    h(1) = bar(0.5,delta.sed_norm_abs(i),      'FaceColor',col1,  'EdgeColor','none'); hold on %[0 0 0]); hold on

    if delta.pval_dt(i) < 0.05 % If p<0.05, fill the flood trend bar fully with colors

        h(2) = bar(1.5,delta.rho_dt_norm_abs(i),   'FaceColor',col2,  'EdgeColor','none'); %[0 0 0]);

    else % For p-value >= 0.05: distinguish between 0.05<=p<0.10 and p>=0.10

        if (delta.pval_dt(i) >= 0.05) && (delta.pval_dt(i) < 0.10) % For "weakly non-significant" trends: divide the bar into 6 almost touching bars
            nb = 6; % number of (almost touching) bars
        elseif (delta.pval_dt(i) >= 0.10) % For "strongly non-significant" trends: divide the bar into 4 bars with quite some space in between
            nb = 4; % number of separated bars
        else
            error('something wrong with p-values')
        end

        h(2) = bar(1.5,delta.rho_dt_norm_abs(i),   'FaceColor','w','EdgeColor','none'); % Put a white big bar underneath
        w = 0.1; % relative width of the individual bars
        bc = linspace(3*w/2, 1-3*w/2, nb); % individual bar center locations
        for nbi = 1:nb % Draw each individual bar
            bci = 1.0+bc(nbi); % Define the center of the currently drawn individual bar
            h(2) = bar(bci,delta.rho_dt_norm_abs(i),w, 'FaceColor',col2,  'EdgeColor','none');
        end
        h(2) = bar(1.5,delta.rho_dt_norm_abs(i),   'FaceColor','none','EdgeColor','none'); %[0 0 0]); % Surround the group of individual thin bars with a black edge

    end

    h(3) = bar(2.5,delta.SLR_norm_abs(i),      'FaceColor',col3,  'EdgeColor','none'); %[0 0 0]);

    yline(0)

    title(delta.plot_name(i))%,'FontSize',fs)

    axis([-0.1 3 1.0*barmin 1.0*barmax])
    set(gca,'XTickLabels',[],'YTickLabels',[])%,'FontSize',fs)

    set(gca,'color','none','box','off','XColor','none','YAxisLocation','right','YTick',[])
    view([90 90])
    pbaspect([1 1 1])

end

% Legend

x_rel = left + 0.4*width;
y_rel = 1.04*bottom;

% Position for text strings in legend
lr = 1.2;
ll = -1.2;
lt = 3.0;

% Create a white rectangle as background
axes('Position',[0.965*x_rel 0.99*y_rel 5.3*barwidth 1.5*barheight],'color','w','XTick',[],'YTick',[],'XColor','none','YColor','none')

% Create the actual legend
axes('Position',[x_rel 1.020*y_rel barwidth barheight])

val = max(delta.sed_norm_abs);
leg_txt = num2str(100*max(delta.sed),'%.0f'); % Express in % in the legend
% cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val))); % Index from the colormap
col1 = col1_pls; % mean(cmap(cmap_ind,:),1); % Corresponding color
h(1) = bar(0.5,val,'FaceColor',col1,'EdgeColor','none'); hold on %[0 0 0]); hold on
text(0.5,lr,leg_txt,'HorizontalAlignment','left','VerticalAlignment','middle')%,'FontSize',fs)

val = -max(delta.sed_norm_abs); %min(delta.sed_norm_abs);
leg_txt = num2str(-100*max(delta.sed),'%.0f'); %num2str(100*min(delta.sed),'%.1f'); % Express in % in the legend
% cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val))); % Index from the colormap
col1 = col1_min; %mean(cmap(cmap_ind,:),1); % Corresponding color
h(1) = bar(0.5,val,'FaceColor',col1,'EdgeColor','none'); %[0 0 0]);
text(0.5,ll,leg_txt,'HorizontalAlignment','right','VerticalAlignment','middle')%,'FontSize',fs)

val = -min(delta.rho_dt_norm_abs); %max(delta.rho_dt_norm_abs);
leg_txt = [num2str(-1000*min(delta.rho_dt),'%.0f')]; %[num2str(1000*max(delta.rho_dt),'%.1f')];
% cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val))); % Index from the colormap
col2 = col2_pls; %mean(cmap(cmap_ind,:),1); % Corresponding color
h(2) = bar(1.5,val,'FaceColor',col2,'EdgeColor','none'); %[0 0 0]);
text(1.5,lr,leg_txt,'HorizontalAlignment','left','VerticalAlignment','middle')%,'FontSize',fs)

val = min(delta.rho_dt_norm_abs);
leg_txt = [num2str(1000*min(delta.rho_dt),'%.0f')];
% cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val))); % Index from the colormap
col2 = col2_min; %mean(cmap(cmap_ind,:),1); % Corresponding color
h(2) = bar(1.5,val,'FaceColor',col2,'EdgeColor','none'); %[0 0 0]);
text(1.5,ll,leg_txt,'HorizontalAlignment','right','VerticalAlignment','middle')%,'FontSize',fs)

val = max(delta.SLR_norm_abs);
leg_txt = num2str(max(delta.SLR),'%.0f');
% cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val))); % Index from the colormap
col3 = col3_pls; %mean(cmap(cmap_ind,:),1); % Corresponding color
h(3) = bar(2.5,val,'FaceColor',col3,'EdgeColor','none'); %[0 0 0]);
text(2.5,lr,leg_txt,'HorizontalAlignment','left','VerticalAlignment','middle')%,'FontSize',fs)

val = -max(delta.SLR_norm_abs); %min(delta.SLR_norm_abs);
leg_txt = num2str(-max(delta.SLR),'%.0f'); %[num2str(min(delta.SLR),'%.1f')];
% cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val))); % Index from the colormap
col3 = col3_min; %mean(cmap(cmap_ind,:),1); % Corresponding color
h(3) = bar(2.5,val,'FaceColor',col3,'EdgeColor','none'); %[0 0 0]);
text(2.5,ll,leg_txt,'HorizontalAlignment','right','VerticalAlignment','middle')%,'FontSize',fs)

leg_sed = 'Fluvial sediment retention [%]';
text(0.5,lt,leg_sed,'HorizontalAlignment','left','VerticalAlignment','middle')%,'FontSize',fs)

leg_flood = ['Trend in surge-discharge correlation [10^{-3} yr^{-1}]'];
text(1.5,lt,leg_flood,'HorizontalAlignment','left','VerticalAlignment','middle')%,'FontSize',fs)

leg_rsl = 'Relative sea-level change [mm yr^{-1}]';
text(2.5,lt,leg_rsl,'HorizontalAlignment','left','VerticalAlignment','middle')%,'FontSize',fs)

yline(0)

title('Legend')%,'FontSize',fs)
axis([-0.1 3 1.0*barmin 1.0*barmax])
set(gca,'XTickLabels',[],'YTickLabels',[])%,'FontSize',fs)

set(gca,'color','none','box','off','XColor','none','YAxisLocation','right','YTick',[])
view([90 90])%% Embed all bar diagrams in a worldmap
pbaspect([1 1 1])

%%%% Create a second legend, to indicate the significance levels in rho_dt
x_rel = left + 0.01*width;
y_rel = 1.04*bottom;

% Position for text strings in legend
lr = 2.7;
ll = -1.5;
lt = 1.2;

% Create a white rectangle as background
axes('Position',[0.995*x_rel 0.99*y_rel 2.2*barwidth 1.5*barheight],'color','w','XTick',[],'YTick',[],'XColor','none','YColor','none')

% Create the actual legend
axes('Position',[x_rel 1.020*y_rel barwidth barheight])

% p<0.05 %%%%%%%%%%%%%%%%%%

val = 0.5; %max(delta.sed_norm_abs);
% cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val))); % Index from the colormap
col1 = col2_pls; %mean(cmap(cmap_ind,:),1); % Corresponding color
h(1) = bar(0.5,val,'FaceColor',col1,'EdgeColor','none'); hold on %[0 0 0]); hold on

val = -0.5; %min(delta.sed_norm_abs);
% cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val))); % Index from the colormap
col1 = col2_min; %mean(cmap(cmap_ind,:),1); % Corresponding color
h(1) = bar(0.5,val,'FaceColor',col1,'EdgeColor','none'); %[0 0 0]);

% 0.10<=p<0.05 %%%%%%%%%%%%%%%%%%

val = 0.5; %max(delta.rho_dt_norm_abs);
% cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val))); % Index from the colormap
col2 = col2_pls; %mean(cmap(cmap_ind,:),1); % Corresponding color
nb = 6;
h(2) = bar(1.5,val,'FaceColor','w','EdgeColor','none'); % Put a white big bar underneath
w = 0.1; % relative width of the individual bars
bc = linspace(3*w/2, 1-3*w/2, nb); % individual bar center locations
for nbi = 1:nb % Draw each individual bar
    bci = 1.0+bc(nbi); % Define the center of the currently drawn individual bar
    h(2) = bar(bci,val,w, 'FaceColor',col2,  'EdgeColor','none');
end
h(2) = bar(1.5,val,   'FaceColor','none','EdgeColor','none'); %[0 0 0]); % Surround the group of individual thin bars with a black edge

val = -0.5; %min(delta.rho_dt_norm_abs);
% cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val))); % Index from the colormap
col2 = col2_min; %mean(cmap(cmap_ind,:),1); % Corresponding color
nb = 6;
h(2) = bar(1.5,val,'FaceColor','w','EdgeColor','none'); % Put a white big bar underneath
w = 0.1; % relative width of the individual bars
bc = linspace(3*w/2, 1-3*w/2, nb); % individual bar center locations
for nbi = 1:nb % Draw each individual bar
    bci = 1.0+bc(nbi); % Define the center of the currently drawn individual bar
    h(2) = bar(bci,val,w, 'FaceColor',col2,  'EdgeColor','none');
end
h(2) = bar(1.5,val,   'FaceColor','none','EdgeColor','none'); %[0 0 0]); % Surround the group of individual thin bars with a black edge

% p > 0.10 %%%%%%%%%%%%%%%%%%

val = 0.5; %max(delta.SLR_norm_abs);
% cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val))); % Index from the colormap
col3 = col2_pls; %mean(cmap(cmap_ind,:),1); % Corresponding color
nb = 4;
h(3) = bar(2.5,val,'FaceColor','w','EdgeColor','none'); % Put a white big bar underneath
w = 0.1; % relative width of the individual bars
bc = linspace(3*w/2, 1-3*w/2, nb); % individual bar center locations
for nbi = 1:nb % Draw each individual bar
    bci = 2.0+bc(nbi); % Define the center of the currently drawn individual bar
    h(3) = bar(bci,val,w, 'FaceColor',col3,  'EdgeColor','none');
end
h(3) = bar(2.5,val,   'FaceColor','none','EdgeColor','none'); %[0 0 0]); % Surround the group of individual thin bars with a black edge

val = -0.5; %min(delta.SLR_norm_abs);
% cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val))); % Index from the colormap
col3 = col2_min; %mean(cmap(cmap_ind,:),1); % Corresponding color
nb = 4;
h(3) = bar(2.5,val,'FaceColor','w','EdgeColor','none'); % Put a white big bar underneath
w = 0.1; % relative width of the individual bars
bc = linspace(3*w/2, 1-3*w/2, nb); % individual bar center locations
for nbi = 1:nb % Draw each individual bar
    bci = 2.0+bc(nbi); % Define the center of the currently drawn individual bar
    h(3) = bar(bci,val,w, 'FaceColor',col3,  'EdgeColor','none');
end
h(3) = bar(2.5,val,   'FaceColor','none','EdgeColor','none'); %[0 0 0]); % Surround the group of individual thin bars with a black edge

% Add labels to this legend
leg_plow = ['p < 0.05'];
text(0.5,lt,leg_plow,'HorizontalAlignment','left','VerticalAlignment','middle')%,'FontSize',fs)

leg_pmid = ['0.05 \leq p < 0.10'];
text(1.5,lt,leg_pmid,'HorizontalAlignment','left','VerticalAlignment','middle')%,'FontSize',fs)

leg_phigh = ['p \geq 0.10'];
text(2.5,lt,leg_phigh,'HorizontalAlignment','left','VerticalAlignment','middle')%,'FontSize',fs)

yline(0)

title('                    Significance of trend')%,']Position',)%,'FontSize',fs)
axis([-0.1 3 1.0*barmin 1.0*barmax])
set(gca,'XTickLabels',[],'YTickLabels',[])%,'FontSize',fs)

set(gca,'color','none','box','off','XColor','none','YAxisLocation','right','YTick',[])
view([90 90])%% Embed all bar diagrams in a worldmap
pbaspect([1 1 1])

% Save plot
    
% fontsize(gcf,fs,"points")
% fontsize(gca,fs,"points")
% fontname(gcf,'Helvetica')
% fontname(gca,'Helvetica')
% set(gcf,'FontWeight', 'Normal')
set(findall(gcf, 'Type', 'Text'),'FontWeight', 'Normal','FontName','Helvetica','FontSize',fs,'units','points')
set(findall(gca, 'Type', 'Text'),'FontWeight', 'Normal','FontName','Helvetica','FontSize',fs,'units','points')

savename = 'Threats_combined_bars_balance_rotated_worldmap_optimized';
dpi = 300;
exportgraphics(gcf,['output/',savename,'.pdf'],'Resolution',dpi,'ContentType','vector')
exportgraphics(gcf,['output/',savename,'.png'],'Resolution',dpi,'ContentType','vector')

% OK.

%% Embed all bar diagrams in a worldmap - optimized legend & longer legends (final)
    
close(figure(20));figure(20)

% Set up plot window size such that the figure itself is 17.8cm wide (= text width)
window_width = 26.74;
window_height = 16.49;

set(gcf,'units','centimeters','outerposition',[2 2 window_width window_height],'color','w')

fs = 8; % FontSize (in points)

pba = [1 2 1]; % pbaspect

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Make a world map with the locations of all selected deltas indicated
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

delta.lon(delta.lon>180) = delta.lon(delta.lon>180) - 360; % Now make sure all longitudes are between -180 and 180 degrees again

ax = worldmap(latlim,lonlim);
setm(ax,'mapprojection','eqdcylin');
land = shaperead('landareas.shp', 'UseGeoCoords', true);
geoshow(ax, land, 'FaceColor', col.gry1,'EdgeColor','none') %[0.9 0.9 0.9]
hold on
scatterm(delta.lat,delta.lon,10,col.gry3,'Filled') %[0.6 0.6 0.6]
    
plabel off
mlabel off
gridm off

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Define the size of each bar plot and define color schemes
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Determine the y-axis range
barmin = min([min(delta.sed_norm_abs),min(delta.rho_dt_norm_abs),min(delta.SLR_norm_abs)]);
barmax = max([max(delta.sed_norm_abs),max(delta.rho_dt_norm_abs),max(delta.SLR_norm_abs)]);

% Choose a color scheme and prepare to color the bars according to value
cmap = cmocean('balance'); % (blue <0, white 0, red >0)
cmap_amp = max([abs(barmin),abs(barmax)]); % Maximal absolute bar value
cmap_val = linspace(-cmap_amp,cmap_amp,length(cmap)); % Bar values that each of the 256 elements in the cmap corresponds to

% Set fixed colors for each of the 3 bars, and distinguish <0 (blue) from >0 (red)
col1_pls = col.red5;
col1_min = col.blu5;
col2_pls = col.red3;
col2_min = col.blu3;
col3_pls = col.red1;
col3_min = col.blu1;

% Manually determine the relative position of the world map compared to the entire plot window
left    = 3.58/19.82;
bottom  = 2.53/10.64;
width   = 13.33/19.82;
height  = 6.23/10.64;

% Set the relative width/height of each bar plot
barwidth = 0.06;
barheight = barwidth;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Make connector lines to connect each barplot to its river delta location
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

for i = 1:length(delta.lat)
    if delta.plot_name(i) == 'Yukon'
        con_length = 7; % Connector length, in degrees
        con_deg = 300; % Angle that connector should make
    elseif delta.plot_name(i) == 'MacKenzie'
        con_length = 7; % Connector length, in degrees
        con_deg = 170; % Angle that connector should make
    elseif delta.plot_name(i) == 'Colorado'
        con_length = 15; % Connector length, in degrees
        con_deg = 150; % Angle that connector should make
    elseif delta.plot_name(i) == 'Mississippi'
        con_length = 15; % Connector length, in degrees
        con_deg = 110; % Angle that connector should make
    elseif delta.plot_name(i) == 'Rio Grande'
        con_length = 25; % Connector length, in degrees
        con_deg = 200; % Angle that connector should make
    elseif delta.plot_name(i) == 'Grijalva'
        con_length = 30; % Connector length, in degrees
        con_deg = 225; % Angle that connector should make
    elseif delta.plot_name(i) == 'Magdalena'
        con_length = 10; % Connector length, in degrees
        con_deg = 60; % Angle that connector should make
    elseif delta.plot_name(i) == 'Orinoco'
        con_length = 28; % Connector length, in degrees
        con_deg = 200; % Angle that connector should make
    elseif delta.plot_name(i) == 'Amazon'
        con_length = 15; % Connector length, in degrees
        con_deg = 215; % Angle that connector should make
    elseif delta.plot_name(i) == 'Sao Francisco'
        con_length = 20; % Connector length, in degrees
        con_deg = 250; % Angle that connector should make
    elseif delta.plot_name(i) == 'Parana'
        con_length = 7; % Connector length, in degrees
        con_deg = 210; % Angle that connector should make
    elseif delta.plot_name(i) == 'Vistula'
        con_length = 16; % Connector length, in degrees
        con_deg = 22; % Angle that connector should make
    elseif delta.plot_name(i) == 'Rhine-Meuse'
        con_length = 18; % Connector length, in degrees
        con_deg = 60; % Angle that connector should make
    elseif delta.plot_name(i) == 'Rhone'
        con_length = 20; % Connector length, in degrees
        con_deg = 120; % Angle that connector should make
    elseif delta.plot_name(i) == 'Po'
        con_length = 6; % Connector length, in degrees
        con_deg = -40; % Angle that connector should make
    elseif delta.plot_name(i) == 'Ebro'
        con_length = 20; % Connector length, in degrees
        con_deg = 165; % Angle that connector should make
    elseif delta.plot_name(i) == 'Moulouya'
        con_length = 18; % Connector length, in degrees
        con_deg = 290; % Angle that connector should make
    elseif delta.plot_name(i) == 'Sebou'
        con_length = 25; % Connector length, in degrees
        con_deg = 190; % Angle that connector should make
    elseif delta.plot_name(i) == 'Senegal'
        con_length = 20; % Connector length, in degrees
        con_deg = 200; % Angle that connector should make
    elseif delta.plot_name(i) == 'Volta'
        con_length = 10; % Connector length, in degrees
        con_deg = 200; % Angle that connector should make
    elseif delta.plot_name(i) == 'Niger'
        con_length = 24 ; % Connector length, in degrees
        con_deg = 245; % Angle that connector should make
    elseif delta.plot_name(i) == 'Congo'
        con_length = 10; % Connector length, in degrees
        con_deg = -30; % Angle that connector should make
    elseif delta.plot_name(i) == 'Limpopo'
        con_length = 17; % Connector length, in degrees
        con_deg = 210; % Angle that connector should make
    elseif delta.plot_name(i) == 'Zambezi'
        con_length = 17; % Connector length, in degrees
        con_deg = 320; % Angle that connector should make
    elseif delta.plot_name(i) == 'Tana'
        con_length = 10; % Connector length, in degrees
        con_deg = -25; % Angle that connector should make
    elseif delta.plot_name(i) == 'Nile'
        con_length = 20; % Connector length, in degrees
        con_deg = 255; % Angle that connector should make
    elseif delta.plot_name(i) == 'Tigris'
        con_length = 12; % Connector length, in degrees
        con_deg = 160; % Angle that connector should make
    elseif delta.plot_name(i) == 'Indus'
        con_length = 20; % Connector length, in degrees
        con_deg = 210; % Angle that connector should make
    elseif delta.plot_name(i) == 'Krishna'
        con_length = 14; % Connector length, in degrees
        con_deg = 220; % Angle that connector should make
    elseif delta.plot_name(i) == 'Godavari'
        con_length = 30; % Connector length, in degrees
        con_deg = 270; % Angle that connector should make
    elseif delta.plot_name(i) == 'Mahanadi'
        con_length = 18; % Connector length, in degrees
        con_deg = 155; % Angle that connector should make
    elseif delta.plot_name(i) == 'Ganges-Brahmaputra'
        con_length = 37; % Connector length, in degrees
        con_deg = 135; % Angle that connector should make
    elseif delta.plot_name(i) == 'Chao Phraya'
        con_length = 34; % Connector length, in degrees
        con_deg = 120; % Angle that connector should make
    elseif delta.plot_name(i) == 'Irriwaddy'
        con_length = 23; % Connector length, in degrees
        con_deg = -78; % Angle that connector should make
    elseif delta.plot_name(i) == 'Mekong'
        con_length = 15; % Connector length, in degrees
        con_deg = -20; % Angle that connector should make
    elseif delta.plot_name(i) == 'Red'
        con_length = 15; % Connector length, in degrees
        con_deg = 105; % Angle that connector should make
    elseif delta.plot_name(i) == 'Mahakam'
        con_length = 20; % Connector length, in degrees
        con_deg = -70; % Angle that connector should make
    elseif delta.plot_name(i) == 'Pearl'
        con_length = 30; % Connector length, in degrees
        con_deg = -10; % Angle that connector should make
    elseif delta.plot_name(i) == 'Yangtze'
        con_length = 22; % Connector length, in degrees
        con_deg = 82; % Angle that connector should make
    elseif delta.plot_name(i) == 'Yellow'
        con_length = 33; % Connector length, in degrees
        con_deg = 130; % Angle that connector should make
    elseif delta.plot_name(i) == 'Han'
        con_length = 10; % Connector length, in degrees
        con_deg = -5; % Angle that connector should make
    elseif delta.plot_name(i) == 'Tone'
        con_length = 22; % Connector length, in degrees
        con_deg = -10; % Angle that connector should make
    elseif delta.plot_name(i) == 'Amur'
        con_length = 20; % Connector length, in degrees
        con_deg = 5; % Angle that connector should make
    elseif delta.plot_name(i) == 'Lena'
        con_length = 15; % Connector length, in degrees
        con_deg = -10; % Angle that connector should make
    elseif delta.plot_name(i) == 'Fly'
        con_length = 20; % Connector length, in degrees
        con_deg = 25; % Angle that connector should make
    elseif delta.plot_name(i) == 'Burdekin'
        con_length = 15; % Connector length, in degrees
        con_deg = -10; % Angle that connector should make
    elseif delta.plot_name(i) == 'Murray'
        con_length = 15; % Connector length, in degrees
        con_deg = -45; % Angle that connector should make
    end
    % Plot the actual connectors
    delta.conlat(i) = delta.lat(i) + sind(con_deg)*con_length;
    delta.conlon(i) = delta.lon(i) + cosd(con_deg)*con_length;
    geoshow([delta.lat(i),delta.conlat(i)]',[delta.lon(i),delta.conlon(i)]','LineStyle','-','Color',col.gry3) %[0.6 0.6 0.6]
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Make a legend that shows the meaning and min/max value of each bar
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

x_rel = left + 0.4*width;
y_rel = 1.04*bottom;

% Position for text strings in legend
lr = 1.2;
ll = -1.2;
lt = 2.0;

% Create a white rectangle as background
axes('Position',[0.965*x_rel 0.99*y_rel 5.3*barwidth 1.5*barheight],'color','w','XTick',[],'YTick',[],'XColor','none','YColor','none')

% Create the actual legend
axes('Position',[x_rel 1.020*y_rel barwidth barheight])

val = max(delta.sed_norm_abs);
leg_txt = num2str(100*max(delta.sed),'%.0f'); % Express in % in the legend
% cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val))); % Index from the colormap
col1 = col1_pls; % mean(cmap(cmap_ind,:),1); % Corresponding color
h(1) = bar(0.5,val,'FaceColor',col1,'EdgeColor','none'); hold on %[0 0 0]); hold on
text(0.5,lr,leg_txt,'HorizontalAlignment','left','VerticalAlignment','middle')%,'FontSize',fs)

val = -max(delta.sed_norm_abs); %min(delta.sed_norm_abs);
leg_txt = num2str(-100*max(delta.sed),'%.0f'); %num2str(100*min(delta.sed),'%.1f'); % Express in % in the legend
% cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val))); % Index from the colormap
col1 = col1_min; %mean(cmap(cmap_ind,:),1); % Corresponding color
h(1) = bar(0.5,val,'FaceColor',col1,'EdgeColor','none'); %[0 0 0]);
text(0.5,ll,leg_txt,'HorizontalAlignment','right','VerticalAlignment','middle')%,'FontSize',fs)

val = -min(delta.rho_dt_norm_abs); %max(delta.rho_dt_norm_abs);
leg_txt = [num2str(-1000*min(delta.rho_dt),'%.0f')]; %[num2str(1000*max(delta.rho_dt),'%.1f')];
% cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val))); % Index from the colormap
col2 = col2_pls; %mean(cmap(cmap_ind,:),1); % Corresponding color
h(2) = bar(1.5,val,'FaceColor',col2,'EdgeColor','none'); %[0 0 0]);
text(1.5,lr,leg_txt,'HorizontalAlignment','left','VerticalAlignment','middle')%,'FontSize',fs)

val = min(delta.rho_dt_norm_abs);
leg_txt = [num2str(1000*min(delta.rho_dt),'%.0f')];
% cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val))); % Index from the colormap
col2 = col2_min; %mean(cmap(cmap_ind,:),1); % Corresponding color
h(2) = bar(1.5,val,'FaceColor',col2,'EdgeColor','none'); %[0 0 0]);
text(1.5,ll,leg_txt,'HorizontalAlignment','right','VerticalAlignment','middle')%,'FontSize',fs)

val = max(delta.SLR_norm_abs);
leg_txt = num2str(max(delta.SLR),'%.0f');
% cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val))); % Index from the colormap
col3 = col3_pls; %mean(cmap(cmap_ind,:),1); % Corresponding color
h(3) = bar(2.5,val,'FaceColor',col3,'EdgeColor','none'); %[0 0 0]);
text(2.5,lr,leg_txt,'HorizontalAlignment','left','VerticalAlignment','middle')%,'FontSize',fs)

val = -max(delta.SLR_norm_abs); %min(delta.SLR_norm_abs);
leg_txt = num2str(-max(delta.SLR),'%.0f'); %[num2str(min(delta.SLR),'%.1f')];
% cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val))); % Index from the colormap
col3 = col3_min; %mean(cmap(cmap_ind,:),1); % Corresponding color
h(3) = bar(2.5,val,'FaceColor',col3,'EdgeColor','none'); %[0 0 0]);
text(2.5,ll,leg_txt,'HorizontalAlignment','right','VerticalAlignment','middle')%,'FontSize',fs)

leg_sed = 'Fluvial sediment retention [%]';
text(0.5,lt,leg_sed,'HorizontalAlignment','left','VerticalAlignment','middle')%,'FontSize',fs)

leg_flood = ['Trend in surge-discharge correlation [10^{-3} yr^{-1}]'];
text(1.4,lt,leg_flood,'HorizontalAlignment','left','VerticalAlignment','middle')%,'FontSize',fs)

leg_rsl = 'Relative sea-level change [mm yr^{-1}]';
text(2.5,lt,leg_rsl,'HorizontalAlignment','left','VerticalAlignment','middle')%,'FontSize',fs)

yline(0)

title('Legend')%,'FontSize',fs)
axis([-0.1 3 1.0*barmin 1.0*barmax])
set(gca,'XTickLabels',[],'YTickLabels',[])%,'FontSize',fs)

set(gca,'color','none','box','off','XColor','none','YAxisLocation','right','YTick',[])
view([90 90])%% Embed all bar diagrams in a worldmap
pbaspect(pba)


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Make a second legend that shows the significance levels for the 2nd bar
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

x_rel = left + 0.01*width;
y_rel = 1.04*bottom;

% Position for text strings in legend
lr = 2.7;
ll = -1.5;
lt = 0.8;

% Create a white rectangle as background
axes('Position',[0.995*x_rel 0.99*y_rel 2.2*barwidth 1.5*barheight],'color','w','XTick',[],'YTick',[],'XColor','none','YColor','none')

% Create the actual legend
axes('Position',[x_rel 1.020*y_rel barwidth barheight])

% p<0.05 %%%%%%%%%%%%%%%%%%

val = 0.5; %max(delta.sed_norm_abs);
% cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val))); % Index from the colormap
col1 = col2_pls; %mean(cmap(cmap_ind,:),1); % Corresponding color
h(1) = bar(0.875,val,'FaceColor',col1,'EdgeColor','none'); hold on %[0 0 0]); hold on

val = -0.5; %min(delta.sed_norm_abs);
% cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val))); % Index from the colormap
col1 = col2_min; %mean(cmap(cmap_ind,:),1); % Corresponding color
h(1) = bar(0.875,val,'FaceColor',col1,'EdgeColor','none'); %[0 0 0]);

% % 0.10<=p<0.05 %%%%%%%%%%%%%%%%%%
% 
% val = 0.5; %max(delta.rho_dt_norm_abs);
% % cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val))); % Index from the colormap
% col2 = col2_pls; %mean(cmap(cmap_ind,:),1); % Corresponding color
% nb = 4;
% h(2) = bar(1.5,val,'FaceColor','w','EdgeColor','none'); % Put a white big bar underneath
% w = 0.1; % relative width of the individual bars
% bc = linspace(3*w/2, 1-3*w/2, nb); % individual bar center locations
% for nbi = 1:nb % Draw each individual bar
%     bci = 1.0+bc(nbi); % Define the center of the currently drawn individual bar
%     h(2) = bar(bci,val,w, 'FaceColor',col2,  'EdgeColor','none');
% end
% h(2) = bar(1.5,val,   'FaceColor','none','EdgeColor','none'); %[0 0 0]); % Surround the group of individual thin bars with a black edge
% 
% val = -0.5; %min(delta.rho_dt_norm_abs);
% % cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val))); % Index from the colormap
% col2 = col2_min; %mean(cmap(cmap_ind,:),1); % Corresponding color
% nb = 4;
% h(2) = bar(1.5,val,'FaceColor','w','EdgeColor','none'); % Put a white big bar underneath
% w = 0.1; % relative width of the individual bars
% bc = linspace(3*w/2, 1-3*w/2, nb); % individual bar center locations
% for nbi = 1:nb % Draw each individual bar
%     bci = 1.0+bc(nbi); % Define the center of the currently drawn individual bar
%     h(2) = bar(bci,val,w, 'FaceColor',col2,  'EdgeColor','none');
% end
% h(2) = bar(1.5,val,   'FaceColor','none','EdgeColor','none'); %[0 0 0]); % Surround the group of individual thin bars with a black edge

% p >= 0.05 %%%%%%%%%%%%%%%%%%

val = 0.5; %max(delta.SLR_norm_abs);
% cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val))); % Index from the colormap
col3 = col2_pls; %mean(cmap(cmap_ind,:),1); % Corresponding color
nb = 4;
h(3) = bar(2.125,val,'FaceColor','w','EdgeColor','none'); % Put a white big bar underneath
w = 0.1; % relative width of the individual bars
bc = linspace(3*w/2, 1-3*w/2, nb); % individual bar center locations
for nbi = 1:nb % Draw each individual bar
    bci = 1.625+bc(nbi); % Define the center of the currently drawn individual bar
    h(3) = bar(bci,val,w, 'FaceColor',col3,  'EdgeColor','none');
end
h(3) = bar(2.125,val,   'FaceColor','none','EdgeColor','none'); %[0 0 0]); % Surround the group of individual thin bars with a black edge

val = -0.5; %min(delta.SLR_norm_abs);
% cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val))); % Index from the colormap
col3 = col2_min; %mean(cmap(cmap_ind,:),1); % Corresponding color
nb = 4;
h(3) = bar(2.125,val,'FaceColor','w','EdgeColor','none'); % Put a white big bar underneath
w = 0.1; % relative width of the individual bars
bc = linspace(3*w/2, 1-3*w/2, nb); % individual bar center locations
for nbi = 1:nb % Draw each individual bar
    bci = 1.625+bc(nbi); % Define the center of the currently drawn individual bar
    h(3) = bar(bci,val,w, 'FaceColor',col3,  'EdgeColor','none');
end
h(3) = bar(2.125,val,   'FaceColor','none','EdgeColor','none'); %[0 0 0]); % Surround the group of individual thin bars with a black edge

% Add labels to this legend
leg_plow = ['p < 0.05'];
text(0.875,lt,leg_plow,'HorizontalAlignment','left','VerticalAlignment','middle')%,'FontSize',fs)

% leg_pmid = ['0.05 \leq p < 0.10'];
% text(1.5,lt,leg_pmid,'HorizontalAlignment','left','VerticalAlignment','middle')%,'FontSize',fs)

leg_phigh = ['p \geq 0.05'];
text(2.125,lt,leg_phigh,'HorizontalAlignment','left','VerticalAlignment','middle')%,'FontSize',fs)

yline(0)

title('             Significance of trend')%,']Position',)%,'FontSize',fs)
axis([-0.1 3 1.0*barmin 1.0*barmax])
set(gca,'XTickLabels',[],'YTickLabels',[])%,'FontSize',fs)

set(gca,'color','none','box','off','XColor','none','YAxisLocation','right','YTick',[])
view([90 90])%% Embed all bar diagrams in a worldmap
pbaspect(pba)


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Plot the bargraphs in the worldmap
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

for i = 1:length(delta.lat) % Loop over all selected deltas
       
    % Determine the relative x and y position of each river delta (that is, convert the (lat,lon) pairs to relative (x,y) coordinates) 
    x_rel = left   + width*  ( (delta.conlon(i) - lonlim(1))/(lonlim(2) - lonlim(1)) );
    y_rel = bottom + height* ( (delta.conlat(i) - latlim(1))/(latlim(2) - latlim(1)) );

    barax(i) = axes('Position',[x_rel-0.5*barwidth y_rel-0.5*barheight barwidth barheight]);

    % For each of the 3 factors (sediment retention, flood correlation trend, RSL), determine the
    % color that should be used to fill the respective bar

    % Sediment retention
    val = delta.sed_norm_abs(i); % Value of this bar
%     cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val))); % Find the index from the colormap that corresponds to this value
%     col1 = mean(cmap(cmap_ind,:),1); % Corresponding color (average in the "1"-direction, in case there are more than one cmap_ind's
    if val < 0
        col1 = col1_min;
    else
        col1 = col1_pls;
    end

    % Idem for flood correlation trend
    val = delta.rho_dt_norm_abs(i);
%     cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val)));
%     col2 = mean(cmap(cmap_ind,:),1);
    if val < 0
        col2 = col2_min;
    else
        col2 = col2_pls;
    end

    % Idem for RSL
    val = delta.SLR_norm_abs(i);
%     cmap_ind = find(abs(val-cmap_val) == min(abs(val-cmap_val)));
%     col3 = mean(cmap(cmap_ind,:),1);
    if val < 0
        col3 = col3_min;
    else
        col3 = col3_pls;
    end

    % Plot the bars

    h(1) = bar(0.5,delta.sed_norm_abs(i),      'FaceColor',col1,  'EdgeColor','none'); hold on %[0 0 0]); hold on

    if delta.pval_dt(i) < 0.05 % If p<0.05, fill the flood trend bar fully with colors

        h(2) = bar(1.5,delta.rho_dt_norm_abs(i),   'FaceColor',col2,  'EdgeColor','none'); %[0 0 0]);

    else % For p-value >= 0.05: distinguish between 0.05<=p<0.10 and p>=0.10

        if (delta.pval_dt(i) >= 0.05) && (delta.pval_dt(i) < 0.10) % For "weakly non-significant" trends: divide the bar into 6 almost touching bars
            nb = 4; % number of (almost touching) bars
        elseif (delta.pval_dt(i) >= 0.10) % For "strongly non-significant" trends: divide the bar into 4 bars with quite some space in between
            nb = 4; % number of separated bars
        else
            error('something wrong with p-values')
        end

        h(2) = bar(1.5,delta.rho_dt_norm_abs(i),   'FaceColor','w','EdgeColor','none'); % Put a white big bar underneath
        w = 0.1; % relative width of the individual bars
        bc = linspace(3*w/2, 1-3*w/2, nb); % individual bar center locations
        for nbi = 1:nb % Draw each individual bar
            bci = 1.0+bc(nbi); % Define the center of the currently drawn individual bar
            h(2) = bar(bci,delta.rho_dt_norm_abs(i),w, 'FaceColor',col2,  'EdgeColor','none');
        end
        h(2) = bar(1.5,delta.rho_dt_norm_abs(i),   'FaceColor','none','EdgeColor','none'); %[0 0 0]); % Surround the group of individual thin bars with a black edge

    end

    h(3) = bar(2.5,delta.SLR_norm_abs(i),      'FaceColor',col3,  'EdgeColor','none'); %[0 0 0]);

    yline(0)

    title(delta.plot_name(i))%,'FontSize',fs)

    axis([-0.1 3 1.0*barmin 1.0*barmax])
    set(gca,'XTickLabels',[],'YTickLabels',[])%,'FontSize',fs)

    set(gca,'color','none','box','off','XColor','none','YAxisLocation','right','YTick',[])
    view([90 90])
    pbaspect(pba)

end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Save the plot
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    
set(findall(gcf, 'Type', 'Text'),'FontWeight', 'Normal','FontName','Helvetica','FontSize',fs,'units','points')
set(findall(gca, 'Type', 'Text'),'FontWeight', 'Normal','FontName','Helvetica','FontSize',fs,'units','points')

savename = 'worldmap_final';
dpi = 500;
exportgraphics(gcf,['output/',savename,'.pdf'],'Resolution',dpi,'ContentType','vector')
exportgraphics(gcf,['output/',savename,'.png'],'Resolution',dpi,'ContentType','vector')

% OK.