
import numpy as np
import matplotlib.pyplot as plt
import math
import os
from scipy.optimize import least_squares

thisdir = os.path.dirname(os.path.realpath(__file__)) 

def arccot(x):
    """
    Computes the inverse cotangent (arccot) of x in radians.

    The arccotangent function is defined as:
        arccot(x) = π/2 - atan(x)

    Parameters:
        x (float): The input value.

    Returns:
        float: The arccotangent of x in radians.

    Special Cases:
        - If x is 0, the result is π/2 (since cotangent of π/2 is 0).
        - For positive x, the result is in the range (0, π/2).
        - For negative x, the result is in the range (π/2, π).
    """
    answer = math.pi / 2 - math.atan(x)
    if answer < math.pi / 2: 
        answer = math.pi - answer
    return answer
def calculate_T(p1, v1, p2, v2, R):
    """
    Calculate the time(s) T when two moving points in 3D space are distance R apart.
    
    Parameters:
    p1, v1: np.array - Initial position and velocity of point 1
    p2, v2: np.array - Initial position and velocity of point 2
    R: float - Desired separation distance
    
    Returns:
    List of valid times T where the distance condition is met.
    """
    dp = p1 - p2
    dv = v1 - v2
    
    a = np.dot(dv, dv)
    b = 2 * np.dot(dp, dv)
    c = np.dot(dp, dp) - R**2
    
    discriminant = b**2 - 4*a*c
    
    if discriminant < 0:
        return []  # No real solutions
    
    sqrt_disc = np.sqrt(discriminant)
    T1 = (-b + sqrt_disc) / (2 * a) if a != 0 else None
    T2 = (-b - sqrt_disc) / (2 * a) if a != 0 else None
    
    return [T for T in [T1, T2] if T is not None]  # Return all valid times, including past times

def calculate_distance(p1, p2):
    return math.sqrt((p2[0] - p1[0]) * (p2[0] - p1[0]) + (p2[1] - p1[1]) * (p2[1] - p1[1]) +(p2[2] - p1[2]) * (p2[2] - p1[2]) )

def calculate_P(T, p1, v1, p2, v2):
    """
    Calculate the positions and velocities of two moving points at time T.
    
    Parameters:
    T: float - The time at which to calculate positions and velocities
    p1, v1: np.array - Initial position and velocity of point 1
    p2, v2: np.array - Initial position and velocity of point 2
    
    Returns:
    Tuple containing the positions and velocities of both points at time T.
    """

    "Correction for gravity required"
    pos1 = p1 + v1 * T
    pos2 = p2 + v2 * T

    vel1 = v1
    vel2 = v2

    # vel1[2] = vel1[2] - 9.81 * T
    # vel2[2] = vel2[2] - 9.81 * T

    return pos1, vel1, pos2, vel2


basedir = thisdir + "/rawdata/nogravityonlyplain_muwet/"
filenames_muwet = []
for i in range(104, 181):
    filename = basedir + "bunke" + str(i) + "/Overal_Drag_Collision.txt"
    # print(filename)
    if (os.path.isfile(filename)):
        Data = np.genfromtxt(filename)
        if (Data.size > 0):
            filenames_muwet.append(filename)

# Check if there are any files to print
# if (len(filenames_muwet) == 0):
#     print("NO FILES FOUND")
# else:
#     print(f"Number of wet files found is {len(filenames_muwet)}")


basedir = thisdir + "/rawdata/nogravityonlyplain_mudry/"
filenames_mudry = []
for i in range(104, 181):
    filename = basedir + "bunke" + str(i) + "/Overal_Drag_Collision.txt"
    # print(filename)
    if (os.path.isfile(filename)):
        Data = np.genfromtxt(filename)
        if (Data.size > 0):
            filenames_mudry.append(filename)

# Check if there are any files to print
# if (len(filenames_mudry) == 0):
#     print("NO FILES FOUND")
# else:
#     print(f"Number of dry files found is {len(filenames_mudry)}")

experiments_bunke_exact_output = np.genfromtxt(thisdir + "/literature/collision_parameters_bunke.csv", delimiter=",", skip_header=1)

bunke_angle = experiments_bunke_exact_output[:,0]
bunke_epsilon = experiments_bunke_exact_output[:,1]
bunke_beta = experiments_bunke_exact_output[:,2]
bunke_mu = experiments_bunke_exact_output[:,3]
bunke_psi1 = experiments_bunke_exact_output[:,4]
bunke_psi2 = experiments_bunke_exact_output[:,5]



# Particle properties
rhol = 945.
mul = 18.9e-3
rhop = 6.e3
dp = 1.5e-3
rp = dp / 2.
ap = 4 * math.pi * rp * rp
vp = (4. / 3.) * math.pi * rp * rp * rp
mp = vp * rhop
layerthickness = 0.15 / 0.75 * rp
gravitational_force = 9.81 * mp
gravitational_force_simulation = np.zeros(len(filenames_muwet))

results = np.zeros((len(filenames_muwet), 7))
qlist = np.zeros((len(filenames_muwet), 1))
vnlist = np.zeros((len(filenames_muwet), 1))
vtlist = np.zeros((len(filenames_muwet), 1))
bridgelength = np.zeros((len(filenames_muwet), 1))

for j in range(len(filenames_muwet)):
    filename = filenames_muwet[j]

    Data = np.genfromtxt(filename)

    before_col = True
    t_merge = 0.
    step_merge = 0

    after_col = False
    t_rupture = 0.
    step_rupture = 0

    for i in range (Data[:, 0].size):
        num_drop = Data[i, 19]
        time = Data[i, 0]
        if (num_drop == 1 and before_col):
            before_col = False
            t_merge = time
            step_merge = i
        
        if (num_drop == 2 and not(before_col) and not(after_col)):
            after_col = True
            t_rupture =  Data[i+1, 0]
            step_rupture = i

    if (after_col == False):
        continue

    Data_before = Data[step_merge, :]
    Data_after = Data[step_rupture, :]

    p1_rupture = Data_after[1:4]
    v1_rupture = Data_after[4:7]
    omega1_rupture = Data_after[7:10]
    p2_rupture = Data_after[10:13]
    v2_rupture = Data_after[13:16]
    omega2_rupture = Data_after[16:19]

    p1_merging = Data_before[1:4]
    v1_merging = Data_before[4:7]
    omega1_merging = Data_before[7:10]
    p2_merging = Data_before[10:13]
    v2_merging = Data_before[13:16]
    omega2_merging = Data_before[16:19]

    
    # The position of the particle is translated back to the collision plane
    deltaTs = calculate_T(p1_rupture, v1_rupture, p2_rupture, v2_rupture, dp)
    if (deltaTs == []):
        continue
    deltaT = max(deltaTs)
    t_after = t_rupture + deltaT
    p1_after, v1_after, p2_after, v2_after = calculate_P(deltaT, p1_rupture, v1_rupture, p2_rupture, v2_rupture)


    # The position of the particle is translated back to the collision plane
    deltaTs = calculate_T(p1_merging, v1_merging, p2_merging, v2_merging, dp)
    deltaT = min(deltaTs)
    t_before = t_merge + deltaT
    p1_before, v1_before, p2_before, v2_before = calculate_P(deltaT, p1_merging, v1_merging, p2_merging, v2_merging)

    # Based on Bunke et al (2024) blz 19
    r_ij = p1_before - p2_before
    n_ij = r_ij / np.linalg.norm(r_ij)
    q = v1_before - v2_before - np.cross(rp * omega1_merging + rp * omega2_merging, n_ij)
    qafter = v1_after - v2_after - np.cross(rp * omega1_rupture + rp * omega2_rupture, n_ij)

    gamma = np.rad2deg(arccot(np.dot(q, n_ij) / np.linalg.norm(np.cross(q,n_ij))))
    gamma_new = np.rad2deg(arccot(np.dot(qafter, n_ij) / np.linalg.norm(np.cross(qafter,n_ij))))

    vnorm = np.dot(q, n_ij) * n_ij
    vtang = q - vnorm
    t_ij = vtang / np.linalg.norm(vtang)

    psi1 = -np.dot(q,t_ij) / np.dot(q,n_ij)
    psi2 = -np.dot(qafter,t_ij) / np.dot(q,n_ij)

    rupture_distance = np.linalg.norm(p1_rupture - p2_rupture)
    bridgelength[j] = rupture_distance - dp

    # exchange momentum is taken as average between both particle before and after velocity
    J = 0.5 * mp * ((v1_before - v1_after) - (v2_before - v2_after))
    beta = -psi2/psi1
    mu = np.linalg.norm(np.cross(n_ij,J)) / np.linalg.norm(np.dot(n_ij,J))
    epsilon = -np.dot(n_ij,qafter) / np.dot(n_ij, q)
    results[j,:] = [gamma, gamma_new, psi1, psi2, epsilon, beta, mu]
    qlist[j] = np.linalg.norm(q)
    vnlist[j] = np.linalg.norm(vnorm)
    vtlist[j] = np.linalg.norm(vtang)

epsilondry = 0.998
epsilonwet = 0.754
betadry = 0.406
betawet = 0.427
mudry = 0.109
muwet = 0.077
gammadry = 151.63
gammawet = 161.78
psi1dry = abs(math.tan(math.radians(gammadry)))
psi1wet = abs(math.tan(math.radians(gammawet)))

results_angle = results[:, 0]
results_psi1 = results[:, 2]
results_psi2 = results[:, 3]
results_epsilon = results[:, 4]
results_beta = results[:, 5]
results_mu = results[:, 6]



# Particle properties
rhol = 945.
mul = 18.9e-3
rhop = 6.e3
dp = 1.5e-3
rp = dp / 2.
ap = 4 * math.pi * rp * rp
vp = (4. / 3.) * math.pi * rp * rp * rp
mp = vp * rhop
layerthickness = 0.15 / 0.75 * rp
gravitational_force = 9.81 * mp
gravitational_force_simulation = np.zeros(len(filenames_mudry))

results_dry = np.zeros((len(filenames_mudry), 7))
qlist_dry = np.zeros((len(filenames_mudry), 1))
vnlist_dry = np.zeros((len(filenames_mudry), 1))
vtlist_dry = np.zeros((len(filenames_mudry), 1))
bridgelength_dry = np.zeros((len(filenames_mudry), 1))

for j in range(len(filenames_mudry)):
    filename = filenames_mudry[j]
    # load input array 
    Data = np.genfromtxt(filename)

    before_col = True
    t_merge = 0.
    step_merge = 0

    after_col = False
    t_rupture = 0.
    step_rupture = 0

    for i in range (Data[:, 0].size):
        num_drop = Data[i, 19]
        time = Data[i, 0]
        if (num_drop == 1 and before_col):
            before_col = False
            t_merge = time
            step_merge = i
        
        if (num_drop == 2 and not(before_col) and not(after_col)):
            after_col = True
            t_rupture =  Data[i+1, 0]
            step_rupture = i

    if (after_col == False):
        continue

    Data_before = Data[step_merge, :]
    Data_after = Data[step_rupture, :]

    p1_rupture = Data_after[1:4]
    v1_rupture = Data_after[4:7]
    omega1_rupture = Data_after[7:10]
    p2_rupture = Data_after[10:13]
    v2_rupture = Data_after[13:16]
    omega2_rupture = Data_after[16:19]

    p1_merging = Data_before[1:4]
    v1_merging = Data_before[4:7]
    omega1_merging = Data_before[7:10]
    p2_merging = Data_before[10:13]
    v2_merging = Data_before[13:16]
    omega2_merging = Data_before[16:19]
   

    # The position of the particle is translated back to the collision plane
    deltaTs = calculate_T(p1_rupture, v1_rupture, p2_rupture, v2_rupture, dp)
    if (deltaTs == []):
        continue
    deltaT = max(deltaTs)
    t_after = t_rupture + deltaT
    p1_after, v1_after, p2_after, v2_after = calculate_P(deltaT, p1_rupture, v1_rupture, p2_rupture, v2_rupture)


    # The position of the particle is translated back to the collision plane
    deltaTs = calculate_T(p1_merging, v1_merging, p2_merging, v2_merging, dp)
    deltaT = min(deltaTs)
    t_before = t_merge + deltaT
    p1_before, v1_before, p2_before, v2_before = calculate_P(deltaT, p1_merging, v1_merging, p2_merging, v2_merging)

    # Based on Bunke et al (2024) blz 19
    r_ij = p1_before - p2_before
    n_ij = r_ij / np.linalg.norm(r_ij)
    q = v1_before - v2_before - np.cross(rp * omega1_merging + rp * omega2_merging, n_ij)
    qafter = v1_after - v2_after - np.cross(rp * omega1_rupture + rp * omega2_rupture, n_ij)

    gamma = np.rad2deg(arccot(np.dot(q, n_ij) / np.linalg.norm(np.cross(q,n_ij))))
    gamma_new = np.rad2deg(arccot(np.dot(qafter, n_ij) / np.linalg.norm(np.cross(qafter,n_ij))))

    vnorm = np.dot(q, n_ij) * n_ij
    vtang = q - vnorm
    t_ij = vtang / np.linalg.norm(vtang)

    psi1 = -np.dot(q,t_ij) / np.dot(q,n_ij)
    psi2 = -np.dot(qafter,t_ij) / np.dot(q,n_ij)

    rupture_distance = np.linalg.norm(p1_rupture - p2_rupture)
    bridgelength_dry[j] = rupture_distance - dp

    # exchange momentum is taken as average between both particle before and after velocity
    J = 0.5 * mp * ((v1_before - v1_after) - (v2_before - v2_after))
    beta = -psi2/psi1
    mu = np.linalg.norm(np.cross(n_ij,J)) / np.linalg.norm(np.dot(n_ij,J))
    epsilon = -np.dot(n_ij,qafter) / np.dot(n_ij, q)
    results_dry[j,:] = [gamma, gamma_new, psi1, psi2, epsilon, beta, mu]
    qlist_dry[j] = np.linalg.norm(q)
    vnlist_dry[j] = np.linalg.norm(vnorm)
    vtlist_dry[j] = np.linalg.norm(vtang)

epsilondry = 0.998
epsilonwet = 0.754
betadry = 0.406
betawet = 0.427
mudry = 0.109
muwet = 0.077
gammadry = 151.63
gammawet = 161.78
psi1dry = abs(math.tan(math.radians(gammadry)))
psi1wet = abs(math.tan(math.radians(gammawet)))


results_angle_dry = results_dry[:, 0]
results_psi1_dry = results_dry[:, 2]
results_psi2_dry = results_dry[:, 3]
results_epsilon_dry = results_dry[:, 4]
results_beta_dry = results_dry[:, 5]
results_mu_dry = results_dry[:, 6]



def arccot(gamma):
    return 0.5 * math.pi - math.atan(gamma)

def modelFunc (gammadeg, epsilon0, beta0, mu0) :
    gamma0rad = arccot((- 2 / 7) * ( 1 + beta0 ) / ((1 + epsilon0) * mu0))
    gamma0deg = math.degrees(gamma0rad)

    epsilon = np.zeros_like(gammadeg)
    beta = np.zeros_like(gammadeg)
    mu = np.zeros_like(gammadeg)
    for i in range(len(gammadeg)):
        gammarad = math.radians(gammadeg[i])
        epsilon[i] = epsilon0
        if gammadeg[i] > gamma0deg:
            beta[i] = beta0
            mu[i] = mu0 * math.tan(gammarad) / math.tan(gamma0rad)
        else:
            beta[i] = (7 * (1 + epsilon0) * mu0) / (2 * abs(math.tan(gammarad))) - 1
            mu[i] = mu0

    return epsilon, beta, mu

def fitFunc(params, gamma, epsilon_val, beta_val, mu_val):
    epsilon0, beta0, mu0 = params
    epsilon, beta, mu = modelFunc(gamma, epsilon0, beta0, mu0)
    return np.concatenate([epsilon - epsilon_val, beta - beta_val, mu - mu_val])




epsilon0new = np.average(results[:,4])
mu0new = np.average(results_mu[0:-6])
# print(mu0new)
p0 = [epsilon0new, betawet, mu0new]

fitParams = least_squares(fitFunc, p0, args=(results_angle[:-1], results_epsilon[:-1], results_beta[:-1], results_mu[:-1]))
epsilon0new =fitParams.x[0]
beta0new =fitParams.x[1]
mu0new =fitParams.x[2]


gamma_model_new = np.linspace(90, 180, 1000)
epsilon_model_new, beta_model_new, mu_model_new = modelFunc(gamma_model_new, epsilon0new, beta0new, mu0new)

print(fitParams)





f, axs = plt.subplots(1,1, figsize=(4,4))
f.subplots_adjust(wspace=.4, hspace=.1)
h = axs.plot(results_angle, results_epsilon, "o", fillstyle='left',zorder=4, color="C0")
h = axs.plot(results_angle_dry, results_epsilon_dry, "o", fillstyle='right',zorder=4, color="C2")
h = axs.plot(bunke_angle, bunke_epsilon, "o", fillstyle='top',zorder=3, color="C1")
h = axs.plot(gamma_model_new, epsilon_model_new, color="C0")
h = axs.plot([90, 180], [0.998, 0.998 ], linestyle=(0, (5,1)), color="C2")
h = axs.plot([90, 180], [0.754, 0.754 ], linestyle=(0, (5,3)), color="C1")

axs.legend(["simulation $\mu_{wet}$", "simulation $\mu_{dry}$", "experiment", "model numerical",  "model exp dry",  "model exp wet"])
axs.set_xlabel(r"$\gamma \; (^o)$")
axs.set_ylabel(r"$\epsilon$ (-)")
axs.set_ylim(0.65, 1.)
axs.set_xlim(90 ,180)
axs.set_xticks((90, 100 ,110, 120 ,130 ,140 ,150 , 160, 170, 180))
f.savefig(thisdir + '/Collision_Epsilon.png', dpi=900, bbox_inches = 'tight')  # Increase dpi for higher resolution



Gammawet = np.linspace(90, gammawet, 1000)
Betawet = np.zeros_like(Gammawet)
for i in range(Betawet.size):
    Betawet[i] = (7 * ( 1 +epsilonwet) * muwet / (2 * abs(math.tan(np.radians(Gammawet[i]))))) - 1
Gammawet = np.append(Gammawet,[180])
Betawet = np.append(Betawet,betawet)
Gammadry = np.linspace(90, gammadry, 1000)
Betadry = np.zeros_like(Gammadry)
for i in range(Betadry.size):
    Betadry[i] = (7 * ( 1 +epsilondry) * mudry / (2 * abs(math.tan(np.radians(Gammadry[i]))))) - 1
Gammadry = np.append(Gammadry,[180])
Betadry = np.append(Betadry,betadry)
f, axs = plt.subplots(1,1, figsize=(4,4))
f.subplots_adjust(wspace=.4, hspace=.1)
h = axs.plot(results_angle, results_beta, "o", fillstyle='left',zorder=4, color = "C0")
h = axs.plot(results_angle_dry, results_beta_dry, "o", fillstyle='right',zorder=4, color="C2")
hh = axs.plot(bunke_angle, bunke_beta, "o", fillstyle='top',zorder=3, color="C1")
h = axs.plot(gamma_model_new, beta_model_new, linestyle=(0, (5,0)), color="C0")
h = axs.plot(Gammadry, Betadry, linestyle=(0, (5,1)), color="C2",zorder=2)
h = axs.plot(Gammawet, Betawet, linestyle=(0, (5,3)), color="C1",zorder=1)

axs.legend(["simulation $\mu_{wet}$", "simulation $\mu_{dry}$", "experiment", "model numerical",  "model exp dry",  "model exp wet"])
axs.set_xlabel(r"$\gamma \; (^o)$")
axs.set_ylabel(r"$\beta$ (-)")
axs.set_ylim(-1, 0.8)

axs.set_xlim(90 ,180)
axs.set_xticks((90, 100 ,110, 120 ,130 ,140 ,150 , 160, 170, 180))

f.savefig(thisdir + '/Collision_Beta.png', dpi=900, bbox_inches = 'tight')  # Increase dpi for higher resolution



f, axs = plt.subplots(1,1, figsize=(4,4))
f.subplots_adjust(wspace=.4, hspace=.1)
h = axs.plot(results_angle, results_mu, "o", fillstyle='left',zorder=4)
h = axs.plot(results_angle_dry, results_mu_dry, "o", fillstyle='right',zorder=4, color="C2")
h = axs.plot(bunke_angle, bunke_mu, "o", fillstyle='top',zorder=3)
h = axs.plot(gamma_model_new, mu_model_new, linestyle=(0, (5,0)), color="C0")
h = axs.plot([90, gammadry, 180], [mudry, mudry, 0], linestyle=(0, (5,1)), color="C2")
h = axs.plot([90, gammawet, 180], [muwet, muwet, 0], linestyle=(0, (5,3)), color="C1")

axs.legend(["simulation $\mu_{wet}$", "simulation $\mu_{dry}$", "experiment", "model numerical",  "model exp dry",  "model exp wet"])
axs.set_xlabel(r"$\gamma \; (^o)$")
axs.set_ylabel(r"$\mu$ (-)")
axs.set_ylim(0, 0.14)
axs.set_xlim(90 ,180)
axs.set_xticks((90, 100 ,110, 120 ,130 ,140 ,150 , 160, 170, 180))

f.savefig(thisdir + '/Collision_Mu.png', dpi=900, bbox_inches = 'tight')  # Increase dpi for higher resolution




Psi1wet = [0, psi1wet, 20]
Psi2wet = [0, (-betawet * psi1wet), (20 - (7./2.) * (1 + epsilonwet) * muwet)]
Psi1dry = [0, psi1dry, 20]
Psi2dry = [0, (-betadry * psi1dry), (20 - (7./2.) * (1 + epsilondry) * mudry)]

gamma0new =  math.degrees(arccot((- 2 / 7) * ( 1 + beta0new ) / ((1 + epsilon0new) * mu0new)))
psi1new = abs(math.tan(math.radians(gamma0new)))
Psi1new = [0, psi1new, 20]
Psi2new = [0, (-beta0new * psi1new), (20 - (7./2.) * (1 + epsilon0new) * mu0new)]

print(f"epsilon0 = {epsilon0new}, beta0 = {beta0new}, mu0 = {mu0new}, gamma0 = {gamma0new}")
f, axs = plt.subplots(1,1, figsize=(5.5,5))
f.subplots_adjust(wspace=.4, hspace=.1)
h = axs.plot(results_psi1, results_psi2, "o", fillstyle='left',zorder=4, color="C0")
h = axs.plot(results_psi1_dry, results_psi2_dry, "o", fillstyle='right',zorder=4, color="C2")
h = axs.plot(bunke_psi1, bunke_psi2, "o", fillstyle='top',zorder=3, color="C1")
h = axs.plot(Psi1new, Psi2new, linestyle=(0, (5,0)), color="C0")
h = axs.plot(Psi1dry, Psi2dry, linestyle=(0, (5,1)), color="C2")
h = axs.plot(Psi1wet, Psi2wet, linestyle=(0, (5,3)), color="C1")

axins = axs.inset_axes(
    [0.1, 0.6, 0.4, 0.35],
    xlim=(0, 0.8), ylim=(-.23, .17))
h = axins.plot(results_psi1, results_psi2, "o", fillstyle='left',zorder=4, color="C0")
h = axins.plot(results_psi1_dry, results_psi2_dry, "o", fillstyle='right',zorder=4, color="C2")
h = axins.plot(bunke_psi1, bunke_psi2, "o", fillstyle='top',zorder=3, color="C1")
h = axins.plot(Psi1new, Psi2new, linestyle=(0, (5,0)), color="C0")
h = axins.plot(Psi1dry, Psi2dry, linestyle=(0, (5,1)), color="C2")
h = axins.plot(Psi1wet, Psi2wet, linestyle=(0, (5,3)), color="C1")

axs.indicate_inset_zoom(axins, edgecolor="black")
axs.legend(["simulation $\mu_{wet}$", "simulation $\mu_{dry}$", "experiment", "model numerical",  "model exp dry",  "model exp wet"] , loc='lower right')
axs.set_xlabel(r"$\Psi_1$ (-)")
axs.set_ylabel(r"$\Psi_2$ (-)")
axs.set_ylim(-0.3, 3.5)
axs.set_xlim(0 ,4)


f.savefig(thisdir + '/Collision_Psi1Psi2.png', dpi=900, bbox_inches = 'tight')  # Increase dpi for higher resolution




