# Quantum Circuit Simulator

This module provides a density matrix based quantum circuit simulator. The simulator gives one the capability to examine
stochastic quantum circuits that are exposed to a wide variety of noise sources, such as Pauli noise and decoherence. 
Concurrency between quantum circuits can be emulated, allowing for the sophisticated administration of decoherence. 
This makes the investigation of distributed quantum systems possible.

The power of this simulator is its ability to calculate the superoperator of a quantum circuit. This superoperator can 
among others be used to efficiently simulate a noisy surface code based on its underlying (distributed) circuits 
[[1]](#1). A forked version of QSurface [[2]](#2), a surface code simulator, supports the use of these 
superoperators for its simulations.

<a id="1">[1]</a> Naomi H. Nickerson, Ying Li, Simon C. Benjamin (2012). *Topological quantum computing with a very 
noisy network and local error rates approaching one percent*.\
<a id="2">[2]</a> Mark Shui Hu (2020). QSurface ([https://github.com/watermarkhu/qsurface]())

## Requirements

* Python 3.8
* Numpy 
* Scipy
* Tqdm
* Varname

## Creating your first concurrent noisy stochastic quantum circuit

To dive straight into the advanced capabilities of the simulator, let our first quantum circuit be the creation of a GHZ
state between four distributed nodes, labelled A, B, C and D. These nodes each contain three qubits of which only one of
them is optically active, enabling it to create remote entanglement. 

Firstly, node A and B create a Bell pair concurrently with the creation of a Bell pair between node C and D. After the 
Bell pairs are created, node A and C perform a round of so-called single selection; an entanglement distillation 
technique with which a noisy Bell pair is consumed in order to raise the fidelity of another noisy Bell pair. 
Effectively, this means that node A and C create a Bell pair after which they use this Bell pair to perform a 
Z-stabilizer measurement on their qubits involved in the Bell pair with the other nodes. The last step is optional and 
depends on the outcome of the stabilizer measurement. If the measurement outcome parity is odd, an X-gate is applied to
the qubits of node C and D that hold the entangled quantum state. If the parity is even, no gates are applied.

We are going to break down the creation of this gate in the different phases. Firstly, we will initialise the quantum 
circuit object such that we have access to the desired nodes

### Initialising the QuantumCircuit object

In a new Python script locate in the root directory of the module we will import the QuantumCircuit object and 
initialise it with the desired nodes. This looks the following:

```python
from circuit_simulation.circuit_simulator import *

# Initialise the QuantumCircuit object with the governing parameters
number_of_qubits = 6            # Each of the 4 nodes has 3 qubits
gate_error = 0.01               # Gate error of 1%
measurement_error = 0.01        # Measurement error of 1%
bell_pair_success_rate = 0.001  # Remote entanglement creation success of 0.1%
bell_pair_noise = 0.05          # Infidelity of Bell pair
T1_idle = 2                     # T1 decoherence time when idle
T2_idle = 2                     # T2 decoherence time when idle

qc = QuantumCircuit(num_qubits=number_of_qubits, init_type=0, noise=True, pg=gate_error, pm=measurement_error,
                    pn=bell_pair_noise, T1_idle=T1_idle, T2_idle=T2_idle, T1_idle_electron=np.inf,
                    T2_idle_electron=np.inf, T1_lde=T1_idle, T2_lde=T2_idle, decoherence=True, probabilistic=True,
                    lde_success=bell_pair_success_rate)

# Define the nodes and assign their qubits
qc.define_node("A", qubits=[5, 4])
qc.define_node("B", qubits=[3])
qc.define_node("C", qubits=[2, 1])
qc.define_node("D", qubits=[0])
```

The `QuantumCircuit` object here has `init_type=0` which means that all qubits are initialised in the zero-state. The 
last four lines introduce the nodes providing their name and the qubit indices that are contained in the node (qubit 
indices start at 0). Now that the QuantumCircuit object is initialised, we can create the actual circuit.

### Creating the Bell pairs
 
Node A and B and node C and D create their Bell pairs concurrently. To enforce this, we need to tell the QuantumCircuit 
object which so-called sub circuits are present with `define_sub_circuit`. The method automatically detects the qubits 
that are involved based on the name. So for example, for `AB` it knows to include the qubits from node A and B. For our
example circuit this looks the following:

```python
qc.define_sub_circuit("ABCD")
qc.define_sub_circuit("AB")
qc.define_sub_circuit("CD", concurrent_sub_circuits="AB")  # Supply the sub circuit it is concurrent with
qc.define_sub_circuit("AC")

qc.define_sub_circuit("A")
qc.define_sub_circuit("B", concurrent_sub_circuits="A")
```

Now that we have defined the sub circuits present, we can start implementing the circuit. The QuantumCircuit needs to 
know for which sub circuit we define the operations with the `start_sub_circuit` method. So for the two sub circuits, 
creating Bell pairs between node A and B and node C and D, this thus looks the following:

```python
qc.start_sub_circuit("AB")
qc.create_bell_pair(5, 3)
qc.start_sub_circuit("CD")
qc.create_bell_pair(2, 0)
```

The creation of remote entanglement is a stochastic process which is taken into account within the `create_bell_pair` 
method.

### Single selection to create GHZ state

Single selection is also a stochastic process not only because of the creation of remote entanglement, but in particular 
because its success is dependent on an even measurement parity. The `single_selection` method returns a boolean that 
indicates whether the distillation was successful or not. Based on this we can determine if the last step, the 
application of additional X-gates is necessary. This results in the following code:

```python
qc.start_sub_circuit("AC")
success = qc.single_selection(CNOT_gate, 4, 1)
if not success:
    qc.start_sub_circuit("A")
    qc.X(2)
    qc.start_sub_circuit("B")
    qc.X(0)
```

### Putting it all together

Putting all this together creates a short protocol that lets one evaluate the creation of a GHZ state between four 
distributed nodes. Decoherence is applied to the qubits according to their idle time, taking the concurrency of the sub
circuits into account. Below one can see the resulting full code:

```python
from circuit_simulation.circuit_simulator import *

# Initialise the QuantumCircuit object with the governing parameters
number_of_qubits = 6            # Each of the 4 nodes has 3 qubits
gate_error = 0.01               # Gate error of 1%
measurement_error = 0.01        # Measurement error of 1%
bell_pair_success_rate = 0.001  # Remote entanglement creation success of 0.1%
bell_pair_noise = 0.05          # Infidelity of Bell pair
T1_idle = 2                     # T1 decoherence time when idle
T2_idle = 2                     # T2 decoherence time when idle

qc = QuantumCircuit(num_qubits=number_of_qubits, init_type=0, noise=True, pg=gate_error, pm=measurement_error,
                    pn=bell_pair_noise, T1_idle=T1_idle, T2_idle=T2_idle, T1_idle_electron=np.inf,
                    T2_idle_electron=np.inf, T1_lde=T1_idle, T2_lde=T2_idle, decoherence=True, probabilistic=True,
                    lde_success=bell_pair_success_rate)

# Define the nodes and their qubits and the sub circuits that are present
qc.define_node("A", qubits=[5, 4])
qc.define_node("B", qubits=[3])
qc.define_node("C", qubits=[2, 1])
qc.define_node("D", qubits=[0])

qc.define_sub_circuit("ABCD")
qc.define_sub_circuit("AB")
qc.define_sub_circuit("CD", concurrent_sub_circuits="AB")  # Supply the sub circuit it is concurrent with
qc.define_sub_circuit("AC")

qc.define_sub_circuit("A")
qc.define_sub_circuit("B", concurrent_sub_circuits="A")

# Create the Bell pairs between the nodes in an concurrent fashion
qc.start_sub_circuit("AB")
qc.create_bell_pair(5, 3)
qc.start_sub_circuit("CD")
qc.create_bell_pair(2, 0)

# Create the GHZ state and apply an additional X-gate based on the measurement parity
qc.start_sub_circuit("AC")
success = qc.single_selection(CNOT_gate, 4, 1)
if not success:
    qc.start_sub_circuit("A")
    qc.X(2)
    qc.start_sub_circuit("B")
    qc.X(0)

# Show the results
qc.end_current_sub_circuit()
qc.draw_circuit(color_nodes=True)  # Displays a schematic representation of the circuit in the console
print(qc.total_density_matrix())  # Shows the (noisy) density matrix of the created GHZ state
```
 
## Stabilizer measurement protocols for a distributed surface code

For more details on the stabilizer measurement protocols and obtaining the corresponding superoperaor, see 
[this readme](stabilizer_measurement_protocols/readme.md).