# Statistical analysis, creation of Figures 6-8

#load packages
library(plyr)
library(dplyr)
library(lme4)
library(lmerTest)
library(Hmisc)
library(corrplot)
library(performance)
library(sjPlot)
library(sjlabelled)
library(sjmisc)
library(ggplot2)
library(grid)

############ A. Set working directories and load full dataset #########

## 1. set working directories (set to own paths)
wd_data  <- "~/Final_repository/Data"
wd_codes    <- "~/Final_repository/Codes"

## 2. full datasets 
setwd(wd_data)
full_dataset <- read.csv('full_dataset.csv')
dataset_dist <- subset(full_dataset, forest_category_2=="disturbed forests")
dataset_undist <- subset(full_dataset, forest_category_2!="disturbed forests")
dataset_AGB_dist <- read.csv('dataset_AGB_dist.csv') #scaled
dataset_Srare_dist <- read.csv('dataset_Srare_dist.csv') #scaled
dataset_relSrare_dist <- read.csv('dataset_relSrare_dist.csv') #scaled
dataset_relative_dist <- read.csv('dataset_relAGB_similarity_dist.csv') #scaled and with adjusted composition similarity values

############ B. Define final models ############

## 1. AGB (from /RO5_Peru/MyCodesRO5/3c. Peru_NFI_finalmodels_and_initial_analysis.R)
AGB_absolute_finalmodel <- lmerTest::lmer(AGB_CWD_ha ~ CWD +slope+nitrogen_0.5cm_mean+tc.5000m+nrst_navigwtrway_road+time_since_disturbance_ln+Absolute_anomaly+(1|new_cod_plot), data=dataset_AGB_dist,REML=FALSE, na.action='na.fail')
plot_model(AGB_absolute_finalmodel, show.values = TRUE)
performance::r2_nakagawa(AGB_absolute_finalmodel)

## 2. AGB (relative) (from /RO5_Peru/MyCodesRO5/3c. Peru_NFI_finalmodels_and_initial_analysis.R)
AGB_relative_finalmodel <- lmerTest::lmer(rel_AGB~ CWD +slope+nitrogen_0.5cm_mean+tc.5000m+nrst_navigwtrway_road+time_since_disturbance_ln+Absolute_anomaly+(1|new_cod_plot), data=dataset_relative_dist,REML=FALSE, na.action='na.fail')
plot_model(AGB_relative_finalmodel, show.values = TRUE)
performance::r2_nakagawa(AGB_relative_finalmodel)

## 3. Rarefied richness (from /RO5_Peru/MyCodesRO5/3c. Peru_NFI_finalmodels_rarefiedrichness.R)
Rrichness_absolute_finalmodel <- lmerTest::lmer(Srare ~ CWD +slope+nitrogen_0.5cm_mean+tc.5000m+nrst_navigwtrway_road+time_since_disturbance+Absolute_anomaly+(1|new_cod_plot), data=dataset_Srare_dist,REML=FALSE, na.action='na.fail')
plot_model(Rrichness_absolute_finalmodel, show.values = TRUE)
performance::r2_nakagawa(Rrichness_absolute_finalmodel)

## 4. Rarefied richness (relative) (from /RO5_Peru/MyCodesRO5/3c. Peru_NFI_finalmodels_rarefiedrichness.R)
Rrichness_relative_finalmodel <- lmerTest::lmer(rel_Srare ~ CWD +slope+nitrogen_0.5cm_mean+tc.5000m+nrst_navigwtrway_road+time_since_disturbance+Absolute_anomaly+(1|new_cod_plot), data=dataset_relSrare_dist,REML=FALSE, na.action='na.fail')
plot_model(Rrichness_relative_finalmodel, show.values = TRUE)
performance::r2_nakagawa(Rrichness_relative_finalmodel)

## 5. Composition similarity in percent to old-growth (from /RO5_Peru/MyCodesRO5/3c. Peru_NFI_finalmodels_and_initial_analysis.R)
Composition_finalmodel <- lmerTest::lmer(adj_similarity_mean_prc ~ CWD +slope+nitrogen_0.5cm_mean+tc.5000m+nrst_navigwtrway_road+time_since_disturbance+Absolute_anomaly+(1|new_cod_plot), data=dataset_relative_dist,REML=FALSE, na.action='na.fail')
plot_model(Composition_finalmodel, show.values = TRUE)
performance::r2_nakagawa(Composition_finalmodel)

############ C. Figures 6 and 7 ############ 

# EFFECT OF TIME SINCE DISTURBANCE and anomalies ON PREDICTED VALUES

### 1. EFFECT OF TIME SINCE DISTURBANCE ###

# 1. function to simulate new dataset 
library(purrr)
library(broom.mixed)

# a.select variables of models
mod_vars_1 <-   c("slope","CWD","tc.5000m","nrst_navigwtrway_road","nitrogen_0.5cm_mean","Absolute_anomaly","time_since_disturbance", "time_since_disturbance_ln")

# b. create function to generate 'new' datasets
preddat_fun = function(data, allvars, var) {
  sums = summarise_at(data,
                      vars( one_of(allvars), -one_of(var) ),
                      median)
  cbind( select_at(data, var), sums)}

# 2. create function to predict new values and plot them, with their uncertainties

effect_of_time <- function(dataset, time_defined, final_model, variable_of_interest, linetype_plot, ribbon_alpha, title_plot, title_y, ylim_min, ylim_max) {
  
  # a. define main variables
  dataset_to_use <- dataset  
  dataset_to_use$final_variable <-  dataset_to_use[,c(variable_of_interest)]
  
  # b. generate 'new' datasets for time since disturbance (linear, log and for composition)
  dataset_for_time <- preddat_fun(dataset_to_use, mod_vars_1, time_defined)
  dataset_for_time$new_cod_plot <- dataset_to_use$new_cod_plot
  dataset_for_time$new_cod_subplot3 <- dataset_to_use$new_cod_subplot3
  
  # c. predict results and CIs
  dataset_for_time[, variable_of_interest] <- predict(final_model,dataset_for_time,re.form=NA)
  mm.a <- model.matrix(terms(final_model),dataset_for_time)
  pvar1 <- diag(mm.a %*% tcrossprod(stats::vcov(final_model),mm.a))
  cmult <- 1.96   #or 1.96, more correct I guess (original was 2)
  dataset_for_time$variable_of_interest <- dataset_for_time[, variable_of_interest]
  dataset_for_time$pred.lower <- dataset_for_time$variable_of_interest - cmult * sqrt(pvar1)
  dataset_for_time$pred.upper <- dataset_for_time$variable_of_interest + cmult * sqrt(pvar1)
  
  # d. plot predictions for disturbed forests
  plot_time <- ggplot(data=dataset_for_time, aes(x=dataset_to_use$time_since_disturbance_unstd, y=variable_of_interest))+
    geom_ribbon(aes(ymin=pred.lower, ymax=pred.upper), linetype=2, alpha=as.numeric(ribbon_alpha), fill="#08306b")+
    geom_point(data=dataset_to_use, aes(x=time_since_disturbance_unstd, y=final_variable), alpha=0.15, size=1, colour="#08306b")+
    geom_line(linetype=linetype_plot, colour="#08306b")+
    #geom_boxplot(data=dataset_undisturbed, aes(x=35, y=variable_of_interest),width=4,outlier.size=0.5,colour="black")+
    labs(title=title_plot ,x="time since disturbance (years)", y=title_y)+
    ylim(as.numeric(c(ylim_min, ylim_max)))+
    theme_light()+
    theme(legend.position='none', axis.text.y = element_blank(),axis.title.y = element_blank())
  
  plot_time
}

# 3. Generate the final plots for time since disturbance
#absolute values
d1 <- effect_of_time(dataset_AGB_dist, 'time_since_disturbance_ln', AGB_absolute_finalmodel, "AGB_CWD_ha", 'solid', "0.2", "(b)", bquote('AGB (Mg '~ha^-1~')'),'0', '900')
d2 <- effect_of_time(dataset_Srare_dist, 'time_since_disturbance', Rrichness_absolute_finalmodel, "Srare", 'solid',"0.2","(d)", 'rarefied richness', '0', '10')

#relative values 
d3 <- effect_of_time(dataset_relative_dist, 'time_since_disturbance_ln', AGB_relative_finalmodel, "rel_AGB", 'blank',"0.0", "(b)", 'AGB (% recovery)', '-50', '685')
d4 <- effect_of_time(dataset_relSrare_dist, 'time_since_disturbance', Rrichness_relative_finalmodel, "rel_Srare",'blank',"0.0","(d)", "tree diversity (% recovery)", '0', '210')
d5 <- effect_of_time(dataset_relative_dist, 'time_since_disturbance', Composition_finalmodel, "adj_similarity_mean_prc", 'blank',"0.0", "(f)", "sp. composition (%r)", '0', '235')

### 2. EFFECT OF DISTURBANCE INTENSITY ###
## EFFECT OF DISTURBANCE INTENSITY (absolute anomaly) ON PREDICTED VALUES 

# 1. function to simulate new datasets already created in section 1


# 2. create function to predict new values and plot them, with their uncertainties

effect_of_intensity <- function(dataset, variable_of_interest, final_model, linetype_plot, ribbon_alpha, title_plot, title_y, ylim_min, ylim_max) {
  
  # a. define main variables
  dataset_to_use <- dataset
  dataset_to_use$final_variable <-  dataset_to_use[,c(variable_of_interest)] 
  
  # b. generate 'new' datasets for time since disturbance (linear, log and for composition)
  dataset_for_anomaly <- preddat_fun(dataset_to_use, mod_vars_1, "Absolute_anomaly")
  dataset_for_anomaly$new_cod_plot <- dataset_to_use$new_cod_plot
  dataset_for_anomaly$new_cod_subplot3 <- dataset_to_use$new_cod_subplot3
  
  # c. predict results and CIs
  dataset_for_anomaly[, variable_of_interest] <- predict(final_model,dataset_for_anomaly,re.form=NA)
  mm.a <- model.matrix(terms(final_model),dataset_for_anomaly)
  pvar1 <- diag(mm.a %*% tcrossprod(stats::vcov(final_model),mm.a))
  cmult <- 1.96   #or 1.96, more correct I guess (original was 2)
  dataset_for_anomaly$variable_of_interest <- dataset_for_anomaly[, variable_of_interest]
  dataset_for_anomaly$pred.lower <- dataset_for_anomaly$variable_of_interest - cmult * sqrt(pvar1)
  dataset_for_anomaly$pred.upper <- dataset_for_anomaly$variable_of_interest + cmult * sqrt(pvar1)
  
  # d. plot predictions for disturbed forests
  
  plot <- ggplot(data=dataset_for_anomaly, aes(x=dataset_to_use$Absolute_anomaly_unstd, y=variable_of_interest))+
    geom_ribbon(aes(ymin=pred.lower, ymax=pred.upper), linetype=2, alpha=as.numeric(ribbon_alpha), fill="#b30000")+
    geom_point(data=dataset_to_use, aes(x=Absolute_anomaly_unstd, y=final_variable), alpha=0.15, size=1, colour="#b30000")+
    geom_line(linetype=linetype_plot, colour="#b30000")+
    #geom_boxplot(data=dataset_undisturbed, aes(x=250, y=variable_of_interest),colour="darkred", width=300, outlier.size=0.5)+
    labs(title = title_plot, x="disturbance intensity (\u0394NDMI)", y=title_y)+
    ylim(as.numeric(c(ylim_min, ylim_max)))+
    theme_light()+
    theme(legend.position='none', axis.text.y = element_text(family='mono'))#+
  #geom_rug(aes(colour="darkorchid3"), alpha=0.1, sides="b")   
  plot}

## 3. Generate the final plots for disturbance intensity
#absolute
i1 <- effect_of_intensity(dataset_AGB_dist, 'AGB_CWD_ha', AGB_absolute_finalmodel,'blank',"0.0","(a)", bquote('AGB (Mg '~ha^-1~')'), '0', '900') # two points are not included in the graph, this is mentioned in the caption
i2 <- effect_of_intensity(dataset_Srare_dist, 'Srare', Rrichness_absolute_finalmodel,'solid',"0.2","(c)", 'Rarefied richness (# spp. per 10 stems)', '0', '10') 

#relative
i3 <- effect_of_intensity(dataset_relative_dist,'rel_AGB', AGB_relative_finalmodel,'solid',"0.2","(a)", 'AGB (%r)', '-60', '685')
i4 <- effect_of_intensity(dataset_relSrare_dist, 'rel_Srare', Rrichness_relative_finalmodel,'solid',"0.2","(c)", "Rarefied richness (%r)", '0', '210')
i5 <- effect_of_intensity(dataset_relative_dist, 'adj_similarity_mean_prc', Composition_finalmodel,'solid',"0.2","(e)", "sp. composition (%r)", '0', '235')

### 4. GENERATE and save FIGURES 4 AND 5 ###

## Figure 6: Effect of time and disturbance intensity on absolute values (AGB and rarefied richness)
cowplot::plot_grid(i1,d1,i2,d2,ncol=2,align="hv")

## Figure 5: Effect of time and disturbance intensity on recovery values (AGB (%r), sp. richness (%r), sp. composition (%r))
cowplot::plot_grid(i3,d3,i4,d4,i5,d5,ncol=2,align="hv")

############ E. CALCULATE INDIVIDUAL PERCENTAGES AND RATES ############ 

# 1. ABSOLUTE AGB AND TIME SINCE DISTURBANCE 
# a.  generate 'new' datasets for time since disturbance (log)
dataset_for_time_ln <- preddat_fun(dataset_AGB_dist, mod_vars_1, 'time_since_disturbance_ln')
dataset_for_time_ln$new_cod_plot <- dataset_AGB_dist$new_cod_plot
dataset_for_time_ln$new_cod_subplot3 <- dataset_AGB_dist$new_cod_subplot3
# b. predict results for AGB_CWD_ha
dataset_for_time_ln$AGB_CWD_ha <-predict(AGB_absolute_finalmodel,dataset_for_time_ln,re.form=NA)
# c. Predict AGB accumulation rates
# first, reassign time since disturbance (unstandardised)
dataset_for_time_ln$time_since_disturbance_unstd <- dataset_dist$time_since_disturbance[match(dataset_for_time_ln$new_cod_subplot3, dataset_AGB_dist$new_cod_subplot3)]

AGB_at_1 <- dataset_for_time_ln$AGB_CWD_ha[dataset_for_time_ln$time_since_disturbance_unstd==1][1]
AGB_at_10 <- dataset_for_time_ln$AGB_CWD_ha[dataset_for_time_ln$time_since_disturbance_unstd==10][1]
AGB_at_11 <- dataset_for_time_ln$AGB_CWD_ha[dataset_for_time_ln$time_since_disturbance_unstd==11][1]
AGB_at_20 <- dataset_for_time_ln$AGB_CWD_ha[dataset_for_time_ln$time_since_disturbance_unstd==20][1]
AGB_at_21 <- dataset_for_time_ln$AGB_CWD_ha[dataset_for_time_ln$time_since_disturbance_unstd==21][1]
AGB_at_30 <- dataset_for_time_ln$AGB_CWD_ha[dataset_for_time_ln$time_since_disturbance_unstd==30][1]
AGB_at_31 <- dataset_for_time_ln$AGB_CWD_ha[dataset_for_time_ln$time_since_disturbance_unstd==31][1]
AGB_at_34 <- dataset_for_time_ln$AGB_CWD_ha[dataset_for_time_ln$time_since_disturbance_unstd==34][1]

#rates
rate_1to10 <- (AGB_at_10-AGB_at_1)/(10-1)# First 10 years
rate_11to20 <- (AGB_at_20-AGB_at_11)/(20-11)# 11-20 years
rate_21to30 <- (AGB_at_30-AGB_at_21)/(30-21)# 21-30 years
rate_31to34 <- (AGB_at_34-AGB_at_31)/(34-31)# 32-34 years
rate_1to20 <- (AGB_at_20-AGB_at_1)/(20-1)# 1to20 years

# Percentage of undisturbed AGB present at 34 years post disturbance
(AGB_at_34/mean(dataset_undist$AGB_CWD_ha))*100

# 2. RAREFIED RICHNESS AND TIME SINCE DISTURBANCE 
# a.  generate 'new' datasets for time since disturbance (log)
dataset_for_time <- preddat_fun(dataset_Srare_dist, mod_vars_1, 'time_since_disturbance')
dataset_for_time$new_cod_plot <- dataset_Srare_dist$new_cod_plot
dataset_for_time$new_cod_subplot3 <- dataset_Srare_dist$new_cod_subplot3
# b. predict results and CIs for AGB_CWD_ha
dataset_for_time$Srare <-predict(Rrichness_absolute_finalmodel,dataset_for_time,re.form=NA)
# c. Predict species richness accumulation rate
# first, reassign time since disturbance (unstandardised)
dataset_for_time$time_since_disturbance_unstd <- dataset_dist$time_since_disturbance[match(dataset_for_time$new_cod_subplot3, dataset_AGB_dist$new_cod_subplot3)]

# d. get rate
Srare_at_1  <- dataset_for_time$Srare[dataset_for_time$time_since_disturbance_unstd==1][1]
Srare_at_34 <- dataset_for_time$Srare[dataset_for_time$time_since_disturbance_unstd==34][1]
(Srare_at_34-Srare_at_1)/(34-1)

# 3. RAREFIED RICHNESS AND  DISTURBANCE INTENSITY
# a.  generate 'new' datasets for time since disturbance
dataset_for_distint_richness <- preddat_fun(dataset_Srare_dist, mod_vars_1, 'Absolute_anomaly')
dataset_for_distint_richness$new_cod_plot <- dataset_Srare_dist$new_cod_plot
dataset_for_distint_richness$new_cod_subplot3 <- dataset_Srare_dist$new_cod_subplot3
# b. predict results and CIs for AGB_CWD_ha
dataset_for_distint_richness$Srare <-predict(Rrichness_absolute_finalmodel,dataset_for_distint_richness,re.form=NA)
# c. Predict AGB accumulation rates
# first, reassign time since disturbance (unstandardised)
dataset_for_distint_richness$Absolute_anomaly_unstd <- full_dataset$Absolute_anomaly[match(dataset_for_distint_richness$new_cod_subplot3, full_dataset$new_cod_subplot3)]
# d. decrease of sp. richness along dist. intensity ranges
#min sp. richness
min(dataset_for_distint_richness$Srare)
#max sp. richness
max(dataset_for_distint_richness$Srare)

# 4. RELATIVE AGB AND  DISTURBANCE INTENSITY
# a.  generate 'new' datasets for time since disturbance (log)
dataset_for_distint_relAGB <- preddat_fun(dataset_relative_dist, mod_vars_1, 'Absolute_anomaly')
dataset_for_distint_relAGB$new_cod_plot <- dataset_relative_dist$new_cod_plot
dataset_for_distint_relAGB$new_cod_subplot3 <- dataset_relative_dist$new_cod_subplot3
# b. predict results and CIs for AGB_CWD_ha
dataset_for_distint_relAGB$rel_AGB <-predict(AGB_relative_finalmodel,dataset_for_distint_relAGB,re.form=NA)
# c. Predict AGB accumulation rates
# first, reassign time since disturbance (unstandardised)
dataset_for_distint_relAGB$Absolute_anomaly_unstd <- full_dataset$Absolute_anomaly[match(dataset_for_distint_relAGB$new_cod_subplot3, full_dataset$new_cod_subplot3)]
# d. min and max 
min(dataset_for_distint_relAGB$rel_AGB)
max(dataset_for_distint_relAGB$rel_AGB)

# 5. RELATIVE RICHNESS AND  DISTURBANCE INTENSITY
# a.  generate 'new' datasets for time since disturbance (log)
dataset_for_distint_relrichness <- preddat_fun(dataset_relSrare_dist, mod_vars_1, 'Absolute_anomaly')
dataset_for_distint_relrichness$new_cod_plot <- dataset_relSrare_dist$new_cod_plot
dataset_for_distint_relrichness$new_cod_subplot3 <- dataset_relSrare_dist$new_cod_subplot3
# b. predict results and CIs for AGB_CWD_ha
dataset_for_distint_relrichness$rel_Srare <-predict(Rrichness_relative_finalmodel,dataset_for_distint_relrichness,re.form=NA)
# c. Predict AGB accumulation rates
# first, reassign time since disturbance (unstandardised)
dataset_for_distint_relrichness$Absolute_anomaly_unstd <- full_dataset$Absolute_anomaly[match(dataset_for_distint_relrichness$new_cod_subplot3, full_dataset$new_cod_subplot3)]
# d. min and max 
min(dataset_for_distint_relrichness$rel_Srare)
max(dataset_for_distint_relrichness$rel_Srare)

# 6. COMPOSITION SIMILARITY AND TIME SINCE DISTURBANCE 
# a.  generate 'new' datasets for time since disturbance (log)
dataset_for_composition <- preddat_fun(dataset_relative_dist, mod_vars_1, 'time_since_disturbance')
dataset_for_composition$new_cod_plot <- dataset_relative_dist$new_cod_plot
dataset_for_composition$new_cod_subplot3 <- dataset_relative_dist$new_cod_subplot3
# b. predict results and CIs for AGB_CWD_ha
dataset_for_composition$similarity_mean <-predict(Composition_finalmodel,dataset_for_composition,re.form=NA)
# c. Predict AGB accumulation rates
# first, reassign time since disturbance (unstandardised)
dataset_for_composition$time_since_disturbance_unstd <- full_dataset$time_since_disturbance[match(dataset_relative_dist$new_cod_subplot3, full_dataset$new_cod_subplot3)]

# d. get rates
Composition_at_1 <- dataset_for_composition$similarity_mean[dataset_for_composition$time_since_disturbance_unstd==1][1]
Composition_at_10 <- dataset_for_composition$similarity_mean[dataset_for_composition$time_since_disturbance_unstd==10][1]
rate_time_composition <- (Composition_at_10-Composition_at_1)/(10-1)
rate_time_composition

#e. min and max 
min(dataset_for_composition$similarity_mean)
max(dataset_for_composition$similarity_mean)

# 7. COMPOSITION SIMILARITY AND DISTURBANCE INTENSITY 
# a.  generate 'new' datasets for time since disturbance (log)
dataset_for_composition <- preddat_fun(dataset_relative_dist, mod_vars_1, 'Absolute_anomaly')
dataset_for_composition$new_cod_plot <- dataset_relative_dist$new_cod_plot
dataset_for_composition$new_cod_subplot3 <- dataset_relative_dist$new_cod_subplot3
# b. predict results and CIs for AGB_CWD_ha
dataset_for_composition$similarity_mean <-predict(Composition_finalmodel,dataset_for_composition,re.form=NA)
# c. Predict AGB accumulation rates
# first, reassign time since disturbance (unstandardised)
dataset_for_composition$Absolute_anomaly_unstd <- full_dataset$Absolute_anomaly[match(dataset_relative_dist$new_cod_subplot3, full_dataset$new_cod_subplot3)]
# d. min and max 
min(dataset_for_composition$similarity_mean)
max(dataset_for_composition$similarity_mean)

############### Figure 8: standardised coefficients ###########################

#Define plot function
plot_all_effects <- function(model, variable_of_interest, title_x, subtitle_x) {
  
  dep_variable <- variable_of_interest
  
  #create results table
  fixed_effects <- as.data.frame(coef(summary(model)))
  fixed_effects$coefficient <- as.factor(rownames(fixed_effects))
  names(fixed_effects)[names(fixed_effects) == "Pr(>|t|)"] <- "p_value"
  fixed_effects$significant <- "no"
  fixed_effects$significant[fixed_effects$p_value<0.05] <- "yes"
  fixed_effects$significant <- as.factor(fixed_effects$significant)
  ci_s <- as.data.frame(confint(model, parm="beta_", level=0.95))
  ci_s$coefficient <- as.factor(rownames(ci_s))
  fixed_effects <- join(fixed_effects, ci_s)
  names(fixed_effects)[names(fixed_effects) == "2.5 %" ]<- "CI_lower"
  names(fixed_effects)[names(fixed_effects) == "97.5 %" ]<- "CI_upper"
  fixed_effects$order <- c(6,5,4,3,2,1,7,8) #order in which the plots should appear (backards order)
  fixed_effects <- subset(fixed_effects, coefficient!="(Intercept)") 
  
  #add labels to new dataframe
  fixed_effects$label[fixed_effects$coefficient=="Absolute_anomaly"] <- "disturbance intensity"
  fixed_effects$label[fixed_effects$coefficient=="time_since_disturbance_ln"] <- "ln(time since disturbance)"
  fixed_effects$label[fixed_effects$coefficient=="time_since_disturbance"] <- "time since disturbance"
  fixed_effects$label[fixed_effects$coefficient=="slope"] <- "slope"
  fixed_effects$label[fixed_effects$coefficient=="CWD"] <- "CWD"
  fixed_effects$label[fixed_effects$coefficient=="nitrogen_0.5cm_mean"] <- "soil N content"
  fixed_effects$label[fixed_effects$coefficient=="tc.5000m"] <- "surrounding tree cover"
  fixed_effects$label[fixed_effects$coefficient=="nrst_navigwtrway_road"] <- "dist. to nearest waterway/road"
  
  # Plotfigure
  plot <- ggplot(fixed_effects, aes(y=reorder(label, order), x=Estimate))+
    geom_vline(xintercept = 0, colour="#A4A4A4", linetype="longdash")+
    geom_errorbarh(aes(xmin = CI_lower, xmax = CI_upper, y=reorder(label, order), height = 0))+
    geom_point(aes(colour=significant), size=2.5)+
    geom_point(shape=1, size=2.5, colour="black")+ 
    labs(title =title_x, subtitle= subtitle_x, x=NULL) + xlab("Coefficient")+
    scale_color_manual(values=c("#FFFFFF", "#000000"))+  
    theme_bw()+
    theme(legend.position='none', 
          axis.title.y = element_blank())
  plot
}

#apply function over the 5 models
#AGB
AGB_abs_plot <- plot_all_effects(AGB_absolute_finalmodel, 'AGB_CWD_ha', '(a)', 'AGB')
AGB_rel_plot <- plot_all_effects(AGB_relative_finalmodel, 'rel_AGB','(b)','AGB (%r)')
AGB_rel_plot <- AGB_rel_plot + theme(legend.position='none', axis.title.y = element_blank(), axis.text.y=element_blank()) # to take y labels out
# Rarefied richness and composition
Richness_abs_plot <- plot_all_effects(Rrichness_absolute_finalmodel, 'Srare','(c)','sp. richness')
Richness_rel_plot <- plot_all_effects(Rrichness_relative_finalmodel, 'rel_Srare','(d)','sp. richness (%r)')
Richness_rel_plot <- Richness_rel_plot + theme(legend.position='none', axis.title.y = element_blank(), axis.text.y=element_blank()) # to take y labels out
Composition_plot <- plot_all_effects(Composition_finalmodel, 'adj_similarity_mean_prc','(e)', 'sp. composition (%r)')
Composition_plot <- Composition_plot + theme(legend.position='none', axis.title.y = element_blank(), axis.text.y=element_blank()) # to take y labels out

blank <- grid.rect(gp=gpar(col="white"))

#Figure_8 
gridExtra::grid.arrange(AGB_abs_plot,AGB_rel_plot,blank,Richness_abs_plot,Richness_rel_plot,Composition_plot,ncol=3, nrow=2, widths=c(0.90, 0.50, 0.50))
