# Analysis on dead trees and stumps

#load packages
library("plyr")
library("dplyr")
library("lme4")
library("ggplot2")
library("sjPlot")

## set working directories (adapt to own path)
wd_data  <- "~/Final_repository/Data"
wd_codes    <- "~/Final_repository/Codes"

# Load dataset
setwd(wd_data)
Deadtrees_plot <- read.csv("dataset_deadtrees.csv")

# For percentage of number of dead trees
fm1.0 <- lmerTest::lmer(perc_dead_trees_n ~ 1 + (1|new_cod_plot), data=Deadtrees_plot ,REML=FALSE, na.action='na.fail')
fm1.1 <- lmerTest::lmer(perc_dead_trees_n ~ forest_category_2 + (1|new_cod_plot), data=Deadtrees_plot ,REML=FALSE, na.action='na.fail')
summary(fm1.1)
AIC(fm1.0, fm1.1)

#For percentage of basal area
fm2.0 <- lmerTest::lmer(perc_dead_trees_BA ~ 1 + (1|new_cod_plot), data=Deadtrees_plot ,REML=FALSE, na.action='na.fail')
fm2.1 <- lmerTest::lmer(perc_dead_trees_BA ~ forest_category_2 + (1|new_cod_plot), data=Deadtrees_plot ,REML=FALSE, na.action='na.fail')
summary(fm2.1)
AIC(fm2.0, fm2.1)

# Final figure for supplementary material 

describing_plot <- function(data, variable_of_interest, title_main, title_y, ylim_min, ylim_max) {
  
  dataset <- as.data.frame(data)
  dataset$final_variable <-  dataset[,c(variable_of_interest)]
  levels(dataset$forest_category_2) <- gsub(" ", "\n", levels(dataset$forest_category_2))
  
  #plot
  plot <- ggplot(data=dataset, aes(x=forest_category_2, y=final_variable)) + 
    geom_violin(trim  = TRUE, fill="gray") +
    geom_boxplot(width=0.1, outlier.size=0.5, notch=FALSE) +
    stat_summary(fun=mean, geom="point", shape=5, size=2) +
    #stat_summary(fun.data=data_summary, col='red')+
    ylim(as.numeric(c(ylim_min, ylim_max))) +
    ylab(title_y) +xlab(NULL)+labs(title= title_main)+
    theme_light()
  plot
}

p1 <- describing_plot(Deadtrees_plot, "perc_dead_trees_n", 'a.', bquote('% of total number of trees and stumps'), '0', '100')
p1_new <- p1 +  annotate("text", x=1.5, y=97, label= "*", size=8)+
  annotate(geom = "segment", x = 1, xend = 2, y = 95, yend = 95, color = "grey", linetype = "dashed", size = 0.5)

p2 <- describing_plot(Deadtrees_plot, "perc_dead_trees_BA", 'b.', bquote('% of total basal area of trees and stumps'), '0', '100')
p2_new <- p2 +  annotate("text", x=1.5, y=97, label= "*", size=8)+
annotate(geom = "segment", x = 1, xend = 2, y = 95, yend = 95, color = "grey", linetype = "dashed", size = 0.5)

## Figure: effect of disturbance on necromass
cowplot::plot_grid(p1_new,p2_new)

### for table
summary(as.factor(Deadtrees_plot$forest_category_2))

# number of plots with no dead trees
nrow(Deadtrees_plot[Deadtrees_plot$deadtrees_total=="0",]) #556
nrow(subset(Deadtrees_plot[Deadtrees_plot$deadtrees_total=="0",], forest_category_2=="disturbed forests")) #99
nrow(subset(Deadtrees_plot[Deadtrees_plot$deadtrees_total=="0",], forest_category_2=="undisturbed forests")) #457

as.data.frame(Deadtrees_plot %>% 
                group_by(forest_category_2) %>% 
                summarise(nplots = n(),
                mean_deadtrees_perc = mean(perc_dead_trees_n, na.rm=T),
                mean_BA_perc = mean(perc_dead_trees_BA, na.rm=T)))
