clear all
close all
clc

% This script analyses the processed bedlevel data from a flume experiment
% where river dunes were generated. The script detects dune crests and
% troughs, after which it calculates dune heights and lengths.

% V1 - 02 May 2025

% Script written by: Sjoukje I. de Lange
% HKV lijn in water, Lelystad, the Netherlands

% This script belongs to the following publication:
% "Bimodality in subaqueous dune height suggests flickering behavior at high flow", published in Nature Communications (2025).
% Sjoukje I. de Lange(1,2,*), Roeland C. van de Vijsel(1), Paul J.J. F. Torfs(3), Nick P. Wallerstein(1), and A.J.F (Ton) Hoitink(1)
% 1 Wageningen University, Department of Environmental Sciences, Hydrology and Environmental Hydraulics Group, Wageningen, the Netherlands
% 2 HKV lijn in water, Lelystad, the Netherlands
% 3 independent researcher
% * Corresponding author: Sjoukje de Lange (delangesjoukje@gmail.com)

%%
% addpath output
datapath = '..\Data\Data_experiments\Seatek\matlab output\';
load([datapath,'20240710_1533 File 1_output.mat'])
modi = [0.3914272; 0.5187109]; %H/h

%%
% Extract data from the timeseries object
elevation_data = data_final0(:,sensor);  % Elevation values
time_data = time_final;       % Corresponding time values

% Define the threshold for filtering crests and troughs
% threshold = 0.25*std(elevation_data);  % Example threshold value (adjust as needed)
threshold = 0.50*std(elevation_data);  % Example threshold value (adjust as needed)

% Find zero-crossings
sign_changes = diff(sign(elevation_data));  % Compute the difference in sign
zero_crossings = find(sign_changes);        % Find indices where sign changes

% Correct the zero-crossing indices to the time vector
zero_crossing_times = time_data(zero_crossings + 1); % +1 because 'diff' reduces length by 1

% Initialize arrays to store filtered crest and trough times and values
crest_times = [];
crest_values = [];
trough_times = [];
trough_values = [];

% Loop through each interval between zero-crossings
for i = 1:length(zero_crossings)-1
    % Define the interval
    interval_start = zero_crossings(i) + 1;
    interval_end = zero_crossings(i+1);
    
    % Extract the data for this interval
    interval_time = time_data(interval_start:interval_end);
    interval_elevation = elevation_data(interval_start:interval_end);
    
    % Find the crest (maximum) in this interval
    [crest_value, crest_index] = max(interval_elevation);
    crest_time = interval_time(crest_index);
    
    % Find the trough (minimum) in this interval
    [trough_value, trough_index] = min(interval_elevation);
    trough_time = interval_time(trough_index);
    
    % Check if the crest and trough meet the threshold criteria
    if crest_value > threshold
        crest_times = [crest_times; crest_time];
        crest_values = [crest_values; crest_value];
    end
    
    if trough_value < -threshold
        trough_times = [trough_times; trough_time];
        trough_values = [trough_values; trough_value];
    end
end

% Calculate dune heights and periods
dune_heights = [];
dune_periods = [];

% Loop through detected crests to calculate dune heights and periods
for i = 1:length(crest_times)-1
    % Find the trough corresponding to the current crest
    current_crest_time = crest_times(i);
    next_crest_time = crest_times(i+1);
    
    % Find the trough that is between the current crest and the next crest
    valid_troughs = trough_times(trough_times > current_crest_time & trough_times < next_crest_time);

    if ~isempty(valid_troughs)
        % Select the first trough in the interval
        current_trough_time = valid_troughs(1);
        % Calculate the height of the dune
        crest_index = find(crest_times == current_crest_time);
        trough_index = find(trough_times == current_trough_time);
        dune_height = crest_values(crest_index) - trough_values(trough_index);
        dune_heights = [dune_heights; dune_height];
        duneheighttime(i) = current_crest_time;
        
        % Calculate the period of the dune
        dune_period = next_crest_time - current_crest_time;
        dune_periods = [dune_periods; dune_period];
    end
end

duneheighttime = duneheighttime(~(duneheighttime==0));

dune_heights_norm = dune_heights./0.15;

%% Plot detected dunes
figure('units','normalized','outerposition',[0 0 1 1]) % Open a screen-sized figure;
subplot (2,1,1), hold on,
plot(time_data, elevation_data, 'k-', 'LineWidth', 1);
plot(crest_times, crest_values, 'ro', 'MarkerSize', 5, 'MarkerFaceColor', 'r', 'DisplayName', 'Crests');
plot(trough_times, trough_values, 'go', 'MarkerSize', 5, 'MarkerFaceColor', 'g', 'DisplayName', 'Troughs');

% Plot horizontal lines for zero-line and thresholds
% yline(0, 'k--', 'Zero Line', 'LineWidth', 1.5); % Zero line
% yline(threshold, 'r--', sprintf('+%.3f', threshold), 'LineWidth', 1.5); % Positive threshold
% yline(-threshold, 'g--', sprintf('-%.3f', threshold), 'LineWidth', 1.5); % Negative threshold

xlabel('time (s)');
ylabel('bed level (m)');
% title('Dunes, Crests, and Troughs in Elevation Data');
legend ('data', 'crests', 'throughs');
grid on;

subplot (2,1,2), hold on,
plot(time_data, elevation_data, 'k-', 'LineWidth', 1);
plot(crest_times, crest_values, 'ro', 'MarkerSize', 8, 'MarkerFaceColor', 'r', 'DisplayName', 'Crests');
plot(trough_times, trough_values, 'go', 'MarkerSize', 8, 'MarkerFaceColor', 'g', 'DisplayName', 'Troughs');

% Plot horizontal lines for zero-line and thresholds
yline(0, 'k--', 'Zero Line', 'LineWidth', 1.5); % Zero line
yline(threshold, 'r--', sprintf('+%.3f', threshold), 'LineWidth', 1.5); % Positive threshold
yline(-threshold, 'g--', sprintf('-%.3f', threshold), 'LineWidth', 1.5); % Negative threshold

xlabel('time (s)');
ylabel('bed level (m)');
xlim ([3.07e4 3.26e4])
grid on;

set(gcf,'Units','Inches');
pos = get(gcf,'Position');
set(gcf,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(gcf,['output/FigureS6.pdf'],'-dpdf','-r500')
% print(gcf,['SI_Figure_6.png'],'-dpng','-r500')

%% plot dune height and lenght over time ------- to do!
figure; subplot (2,1,1), hold on,
plot(duneheighttime./60, dune_heights, 'k')
yline (modi(1)*0.15); yline (modi(2)*0.15); yline (mean(dune_heights), '--')
ylabel ('\Delta [m]'); xlim ([0 max(duneheighttime)/60])

subplot (2,1,2), hold on,
plot(duneheighttime./60, dune_periods, 'k')
yline (mean(dune_periods), '--')
ylabel ('\lambda [s]'); xlabel ('time [min]')
xlim ([0 max(duneheighttime)/60])

set(gcf,'Units','Inches');
pos = get(gcf,'Position');
set(gcf,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(gcf,['output/FigureS8.pdf'],'-dpdf','-r500')
% print(gcf,['SI_Figure_8.png'],'-dpng','-r500')

%% plot histograms
figure('units','normalized','outerposition',[0.1 0.1 0.5 0.8]) % Open a screen-sized figure;
hold on,

nbins = 20; 
subplot(2,1,1)
yyaxis left
histogram(dune_periods,nbins)
xlabel('\lambda [s]'); ylabel ('frequency')
xlim([min(dune_periods) max(dune_periods)])

yyaxis right
% histogram(wavelet.cwt_max_periods)
plot(seconds(wavelet.period), wavelet.tavgp, 'r')

ylabel ('power');
% ylim ([0.8e-5 3e-5])
ax = gca;
ax.YAxis(1).Color = 'k';
ax.YAxis(2).Color = 'k';
% %detected modi
% xline(184.0546, 'r')
% xline(279.1444, 'r')
% xline(158.2676, 'r')

  


subplot(2,1,2)
histogram(dune_heights,nbins)
set(gca,'XLim',[min(dune_heights) max(dune_heights)])
xlabel('\Delta [m]'); ylabel ('frequency')
xline (modi(1)*0.15, 'r', 'LineWidth', 2); xline (modi(2)*0.15, 'r', 'LineWidth', 2);


set(gcf,'Units','Inches');
pos = get(gcf,'Position');
set(gcf,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(gcf,['output/FigureS9.pdf'],'-dpdf','-r500')
% print(gcf,['SI_Figure_9.png'],'-dpng','-r500')

%% 
% celerity = (flume.x_sensors(1)-flume.x_sensors(end)) / (20*60); % [m/s];
% dune_lengths = dune_periods*celerity;
% 
% X = dune_lengths;
% Y = dune_heights;
% 
% nbins = 20;
% 
% % Define the custom model
% model = @(b, X) b(1) * X .^ b(2);  % Model: Y = a * X^b
% 
% % Define the error function for nonlinear fitting
% error_func = @(b) sum((Y - model(b, X)).^2);  % Sum of squared errors
% 
% % Initial guess for parameters [a, b]
% initial_guess = [1, 1];
% 
% % Perform the nonlinear fit using nlinfit
% b_est = nlinfit(X, Y, model, initial_guess);
% 
% % Extract fitted parameters
% a_fit = b_est(1);
% b_fit = b_est(2);
% 
% % Generate values for the fitted curve
% X_fit = linspace(min(X), max(X), 100);
% Y_fit = model(b_est, X_fit);
% 
% % Calculate R^2 value
% Y_pred = model(b_est, X);
% SS_res = sum((Y - Y_pred).^2);
% SS_tot = sum((Y - mean(Y)).^2);
% R2 = 1 - (SS_res / SS_tot);
% 
% % Emperical relationship (Venditti review paper)
% L_theory = linspace(min(dune_lengths),max(dune_lengths),20);
% H_theory = 0.0677*L_theory.^0.8098;


% Plot the data and the fit
% figure('units','normalized','outerposition',[0 0 1 1]) % Open a screen-sized figure;
% 
% 
% subplot(2,2,3)
% scatter(dune_periods,dune_heights)
% xlabel('dune periods [s]')
% ylabel('dune heights [m]')
% axis([min(dune_periods) max(dune_periods) min(dune_heights) max(dune_heights)])
% 
% subplot(2,2,1)
% histogram(dune_periods,nbins)
% xlabel('dune periods [s]')
% set(gca,'XLim',[min(dune_periods) max(dune_periods)])
% 
% subplot(2,2,4)
% histogram(dune_heights,nbins)
% xlabel('dune heights [m]')
% set(gca,'XLim',[min(dune_heights) max(dune_heights)])
% % set(gca,'view',[90 -90])
% 
% subplot(2,2,2)
% scatter(X, Y, 'DisplayName', 'Data');
% hold on;
% plot(X_fit, Y_fit, 'r-', 'LineWidth', 2, 'DisplayName', sprintf('Fit: Y = %.2f * X^{%.2f}', a_fit, b_fit));
% title(sprintf('Fit: Y = %.2f * X^{%.2f}; R^2 = %.2f', a_fit, b_fit,R2))
% plot(L_theory,H_theory,'-k')
% legend('data','fit','theory','Location','SouthEast')
% 
% % legend;
% 
% % Display R^2 value on the plot
% % text(mean(X), max(Y) - 0.5*range(Y), sprintf('R^2 = %.2f', R2), ...
% %     'HorizontalAlignment', 'center', 'FontSize', 12, 'BackgroundColor', 'w');
% hold off;
% % legend;
% ylabel('dune heights [m]')
% xlabel('dune lengths [m]')
% axis([min(dune_lengths) max(dune_lengths) min(dune_heights) max(dune_heights)])
