%% figure_rawdata_V2.m

% This script compares our own experimental data (river dunes generated in
% a flume experiment) with flume data of Bradley & Venditti (2019) and
% Venditti et al. (2016), and compares these experimental data with a
% theoretical dune height curve.

% V2 - 04 Feb 2025

% Script written by: Sjoukje I. de Lange
% HKV lijn in water, Lelystad, the Netherlands

% This script belongs to the following publication:
% "Bimodality in subaqueous dune height suggests flickering behavior at high flow", published in Nature Communications (2025).
% Sjoukje I. de Lange(1,2,*), Roeland C. van de Vijsel(1), Paul J.J. F. Torfs(3), Nick P. Wallerstein(1), and A.J.F (Ton) Hoitink(1)
% 1 Wageningen University, Department of Environmental Sciences, Hydrology and Environmental Hydraulics Group, Wageningen, the Netherlands
% 2 HKV lijn in water, Lelystad, the Netherlands
% 3 independent researcher
% * Corresponding author: Sjoukje de Lange (delangesjoukje@gmail.com)

clear all
close all
clc

colb = ['k', 'b', 'g', 'y', 'r', 'k', 'b', 'g', 'y', 'r', 'k', 'b']; %divided in threhold-bed-mixlow-mixup-susp (bradley)
col = ['r', 'r', 'b', 'b', 'g','r', 'r', 'b', 'b', 'g','r', 'r']; %divided in bed-mix-sus (delange)
cl = ['r', 'b', 'g']';

cmap = cmocean('thermal');
col_i = round(linspace(0.15*length(cmap),0.95*length(cmap),5));
colors = cmap(col_i,:);

colBB = colors(1,:); 
colBn = colors(2,:);
colVe = colors(3,:);
colBr = colors(4,:);
colLV = colors(5,:); % de Lange & van de Vijsel et al. (this paper)

% OK.

%% add data of own experiment

pathName1 = '..\Data\Data_experiments\Seatek\matlab output\';
fileName1 = [pathName1,'duneheight_experiment.mat'];

addpath(pathName1)
load(fileName1)

modi_exp = modi; clear modi

% OK.

%% Load data from Bradley 2019

pathName2 = '..\Data\Bradley2019\';
fileName2 = [pathName2,'Time_Series_Data.xlsx'];

addpath(pathName2)

[~,sheet_name]=xlsfinfo(fileName2);
opts = detectImportOptions(fileName2, 'NumHeaderLines', 3);
for k=1:numel(sheet_name)
  dataBr{k}=rows2vars(readtable(fileName2, opts, 'Sheet', sheet_name{k}));
end
clear opts

% OK.

%% load venditti

pathName3 = '..\Data\Venditti2016\';
fileName3 = [pathName3,'data_venditti_matlab.xlsx'];

addpath(pathName3)

[~,sheet_name]=xlsfinfo(fileName3);
opts = detectImportOptions(fileName3, 'NumHeaderLines', 3);
for k=1:numel(sheet_name)
  dataVe{k}=readtable(fileName3, opts, 'Sheet', sheet_name{k});
end

% OK.

%% theoretical curve Venditti & Bradley 2021

shsctst = 1:1:100;
Hhtst = -0.001 .* (shsctst-17.7).^2 + 0.417; %for lab

% OK.

%% load modi as determined by Paul

pathName4 = 'input\';
fileName4 = [pathName4,'modi_distance.xlsx'];

addpath(pathName4)

% modi = readtable ('M:\PhDSjoukje\Project Flickering\data\modi_distance.xlsx', 'Sheet', 'duneheight over average depth');
modi = readtable(fileName4, 'Sheet', 'duneheight over average depth');

% OK.

%% Load time series (dune height vs. time) from V2016 and BV2019

pathName5 = 'input\';
fileName5 = [pathName5,'data_av.xlsx'];

addpath(pathName5)

% dat = readtable ('M:\PhDSjoukje\Project Flickering\data\ForRoeland\data_av.xlsx', 'Sheet', 'duneheight over averagedepth', 'HeaderLines', 4);
% T = readtable ('M:\PhDSjoukje\Project Flickering\data\ForRoeland\data_av.xlsx', 'Sheet', 'duneheight over averagedepth','Range','A3:O3'); 
% h= readtable ('M:\PhDSjoukje\Project Flickering\data\ForRoeland\data_av.xlsx', 'Sheet', 'duneheight over averagedepth','Range','A4:O4'); 
dat = readtable(fileName5, 'Sheet', 'duneheight over averagedepth', 'HeaderLines', 4);
T   = readtable(fileName5, 'Sheet', 'duneheight over averagedepth', 'Range', 'A3:O3'); 
h   = readtable(fileName5, 'Sheet', 'duneheight over averagedepth', 'Range', 'A4:O4'); 
T   = rows2vars(T); h = rows2vars(h);

% OK.

%% organise

%organise bradleys data
for k = 1:length(dataBr)
sdHbr(k) = std(cell2mat(dataBr{k}.Var17(32:end))); %std H
mnHbr(k) = nanmean(cell2mat(dataBr{k}.Var17(32:end))); %mean H


% sdHhbr(k) = nanstd(cell2mat(dataBr{k}.Var23(32:end))); %std H/h
% mnHhbr(k) = nanmean(cell2mat(dataBr{k}.Var23(32:end))); %mean H/h
% Hhbr{k} = cell2mat(dataBr{k}.Var23(32:end));
% mnhbr(k) = nanmean(cell2mat(dataBr{k}.Var7(32:end))); %mean h
sdHhbr(k) = nanstd(cell2mat(dataBr{k}.Var21(32:end))); %std H/h
mnHhbr(k) = nanmean(cell2mat(dataBr{k}.Var21(32:end))); %mean H/h
Hhbr{k} = cell2mat(dataBr{k}.Var21(32:end));
mnhbr(k) = nanmean(cell2mat(dataBr{k}.Var5(32:end))); %mean h

mnshscBr(k) = nanmean(cell2mat(dataBr{k}.Var10(32:end))); %mean sh/sc
sdshscBr(k) = nanstd(cell2mat(dataBr{k}.Var10(32:end))); %mean sh/sc
shscBr{k} = cell2mat(dataBr{k}.Var10(32:end)); %mean sh/sc
end

clear idx
%organise vendittis data
for k = 1:length(dataVe)
     [~, idx{k}] = min(abs(dataVe{k}.Var1-10)); %do not use data of first 10 hours

    sdHVe(k) = nanstd(dataVe{k}.Var8(idx{k}:end)); %std H
    mnHVe(k) = nanmean(dataVe{k}.Var8(idx{k}:end)); %mean H

    mnHhVe(k) = nanmean(dataVe{k}.Var8(idx{k}:end)./dataVe{k}.Var3(idx{k}:end)); %mean H/h
    sdHhVe(k) = nanstd(dataVe{k}.Var8(idx{k}:end)./dataVe{k}.Var3(idx{k}:end)); %std H/h
    HhVe{k} = dataVe{k}.Var8(idx{k}:end)./dataVe{k}.Var3(idx{k}:end);
    mnhVe(k) = nanmean(dataVe{k}.Var3(idx{k}:end));   %mean h

    mnshscVe(k) = nanmean(dataVe{k}.Var6(idx{k}:end)); %mean sh/sc
    sdshscVe(k) = nanstd(dataVe{k}.Var6(idx{k}:end)); %mean sh/sc
    
end

%% Extra datasets: Bennett1998

mnHBn = 2.5*10^-3; % [m] mean dune height H
sdHBn = 1.3*10^-3; % [m] std in H
mnhBn = 0.11;      % [m] mean water depth h
mnHhBn = mnHBn/mnhBn; % [-] mean H/h
sdHhBn = sdHBn/mnhBn; % [-] std in H/h

rho_s = 2650; % [kg m^-3] Sediment density
rho_w = 1000; % [kg m^-3] Water density
g = 9.81; % [m s^-2] Gravitational acceleration
nu = 1.05*10^-6; % [m^2 s^-1] Kinematic viscosity of water (at 18 deg C)    
D_50 = 256*10^-6; % [m] Median grain size
tau = 2.63; % [N m^-2] Shear stress
rho_r = (rho_s - rho_w)/rho_w; % [-] Relative submerged density
Re_p = (D_50^(3/2))*sqrt(rho_r*g)/nu; % [-] Particle Reynolds number        
theta = tau/((rho_s - rho_w)*g*D_50); % [-] Shields stress
theta_c = 0.5*(0.22*Re_p^(-0.6) + 0.06*10^(-7.7*Re_p^(-0.6))); % [-] Critical Shields stress

mnshBn = theta; % Shields stress [-]
mnscBn = theta_c; % Critical Shields stress [-]
mnshscBn = mnshBn/mnscBn; % sh/sc
sdshscBn = 0; % No std given

% OK.

%% Extra datasets: Best & Bridge 1992
mnhBB = 0.105;      % [m] mean water depth h

HBB = [5.9, 4.1, 2.9, 3.3, 2.8, 5.8]*10^-3; % [m] dune heights H
mnHBB = mean(HBB); % [m] mean dune height H
sdHBB = std(HBB); % [m] std in H

mnHhBB = mnHBB/mnhBB; % [-] mean H/h
sdHhBB = sdHBB/mnhBB; % [-] std in H/h

shBB = [0.62, 0.56, 0.81, 0.73, 0.87, 0.65]; % [-] Shields stresses
mnshBB = mean(shBB); % [-] mean sh
sdshBB = std(shBB); % [-] std in sh

rho_s = 2650; % [kg m^-3] Sediment density
rho_w = 1000; % [kg m^-3] Water density
g = 9.81; % [m s^-2] Gravitational acceleration
nu = 1.05*10^-6; % [m^2 s^-1] Kinematic viscosity of water (at 18 deg C)    
D_50 = 290*10^-6; % [m] Median grain size
rho_r = (rho_s - rho_w)/rho_w; % [-] Relative submerged density
Re_p = (D_50^(3/2))*sqrt(rho_r*g)/nu; % [-] Particle Reynolds number        
theta_c = 0.5*(0.22*Re_p^(-0.6) + 0.06*10^(-7.7*Re_p^(-0.6))); % [-] Critical Shields stress

mnscBB = theta_c; % sc
mnshscBB = mnshBB/mnscBB; % mean sh/sc
sdshscBB = sdshBB/mnscBB; % std in sh/sc

% OK.

%% Extra datasets: Naqshband (don't show - not comparable: polystyrene)

mnhNa  = [0.15	0.14	0.15	0.16	0.15	0.14	0.15	0.16	0.14]; % [m] mean water depth h
mnshNa = [0.10	0.27	0.83	1.86	2.91	4.97	7.03	9.70	12.1]; % Shields stress [-]

mnsdHhNa = [...
    0.10, 0.03	
    0.13, 0.04	
    0.26, 0.07	
    0.29, 0.08	
    0.26, 0.11	
    0.27, 0.06	
    0.27, 0.06	
    0.18, 0.02	
    0.09, 0.02  ]'; % mean and std in H/h

mnHhNa = mnsdHhNa(1,:); % [-] mean H/h
sdHhNa = mnsdHhNa(2,:); % [-] std in H/h

rho_s = 1055; % [kg m^-3] Sediment density (polystyrene)
rho_w = 1000; % [kg m^-3] Water density
g = 9.81; % [m s^-2] Gravitational acceleration
nu = 1.05*10^-6; % [m^2 s^-1] Kinematic viscosity of water (at 18 deg C)    
D_50 = 2.1*10^-3; % [m] Median grain size
rho_r = (rho_s - rho_w)/rho_w; % [-] Relative submerged density
Re_p = (D_50^(3/2))*sqrt(rho_r*g)/nu; % [-] Particle Reynolds number        
theta_c = 0.5*(0.22*Re_p^(-0.6) + 0.06*10^(-7.7*Re_p^(-0.6))); % [-] Critical Shields stress

mnscNa = theta_c; % Critical Shields stress [-]
mnshscNa = mnshNa/mnscNa; % sh/sc
sdshscNa = 0; % No std given

% OK.

%% significance distance between modi

% %method 1: spread of entire dataset
% sig = sortmodi.m1_m2 ./  (max(max(sortdat)) - min(min(sortdat)))
% sig(isnan(sortmodi.m2)) = NaN;
% thres = 1/3;

%method 2: spread of single experiment
% sig = sortmodi.m1_m2 ./  (max(sortdat) - min(sortdat))'
%thres = 0.15
%method 3: coefficient of variaton
[sortallT, x] = sort(T.Var1);
sortmodi = sortrows(modi, 'T_gem');

mn = nanmean(table2array(dat));
stdd = nanstd(table2array(dat));
CV = stdd./mn;
sig = sortmodi.m1_m2 - CV';
% sig = sortmodi.m1_m2 - 2*CV';

thres = 0;

% Also determine the CV and significance (d_modi / CV) of our own experiment
mnLV = mean(dune_heights)/0.15; % mean H/h (own experiment)
stddLV = std(dune_heights/0.15); % std H/h (own experiment)
CVLV = stddLV./mnLV; % CV
sigLV = (modi_exp(2) - modi_exp(1)) - CVLV'; % significance


close(figure(1)); figure(1)
hold on,
for i = 1:length(sig)
if isnan(sortmodi.m2(i))
    continue

else
    if sig(i)<thres
    scatter (sortmodi.T_gem(i), sortmodi.m1(i), 'k')
    scatter (sortmodi.T_gem(i), sortmodi.m2(i), 'k')
    else
    scatter (sortmodi.T_gem(i), sortmodi.m1(i), 'k', 'filled')
    scatter (sortmodi.T_gem(i), sortmodi.m2(i), 'k', 'filled')
    end
end

end


%% raw data (figure 1)

close(figure(2)); figure(2)
set(gcf,'Position',[50 50 1200 500])
subplot (3,2,1), hold on,
i=1;
plot(cell2mat(dataBr{i}.Var1(2:end))-min(cell2mat(dataBr{i}.Var1(2:end))),cell2mat(dataBr{i}.Var17(2:end))', 'k', 'LineWidth',1.5)
ylim ([0.03 0.1]); xlim ([0 10]);
text (0.75, 0.9, strcat('\theta / \theta_c = ', num2str( round(nanmean(cell2mat(dataBr{i}.Var10(2:end))),1))), 'Units','normalized', 'FontSize', 12)
text (0.05, 0.9, 'a)', 'Units','normalized', 'FontSize', 12)
%xlabel ('time (h)');
ylabel ('\Delta [m]', 'FontSize', 12);
ax = gca; ax.FontSize = 12;

subplot (3,2,3), hold on,
i=3;
plot(cell2mat(dataBr{i}.Var1(2:end))-min(cell2mat(dataBr{i}.Var1(2:end))),cell2mat(dataBr{i}.Var17(2:end)),'k', 'LineWidth',1.5)
ylim ([0.03 0.1]); xlim ([0 10]);
text (0.75, 0.9, strcat('\theta / \theta_c = ', num2str( round(nanmean(cell2mat(dataBr{i}.Var10(2:end))),1))), 'Units','normalized', 'FontSize', 12)
text (0.05, 0.9, 'b)', 'Units','normalized', 'FontSize', 12)
%xlabel ('time (h)');
ylabel ('\Delta [m]', 'FontSize', 12);
ax = gca; ax.FontSize = 12;

% subplot (3,2,5), hold on,%own experiment
% plot(duneheighttime./3600, dune_heights, 'k')
% xlim ([0.03 0.1]); xlim ([0 10]);
% text (0.75, 0.9, strcat('\theta / \theta_c = ', num2str(25)), 'Units','normalized', 'FontSize', 12)
% text (0.05, 0.9, 'b)', 'Units','normalized', 'FontSize', 12)
% %xlabel ('time (h)');
% ylabel ('\Delta [m]', 'FontSize', 12);
% ax = gca; ax.FontSize = 12;

subplot (3,2,5), hold on,
i = 10;
    plot(cell2mat(dataBr{i}.Var1(2:end))-min(cell2mat(dataBr{i}.Var1(2:end))),cell2mat(dataBr{i}.Var17(2:end)),'k' , 'LineWidth',1.5)
xlabel ('time [h]', 'FontSize', 12);ylabel ('\Delta [m]', 'FontSize', 12);
ylim ([0.03 0.1]); xlim ([0 10]);
text (0.75, 0.9, strcat('\theta / \theta_c = ', num2str( round(nanmean(cell2mat(dataBr{i}.Var10(2:end))),1))), 'Units','normalized', 'FontSize', 12)
text (0.05, 0.9, 'c)', 'Units','normalized', 'FontSize', 12)
ax = gca; ax.FontSize = 12;




subplot(3,2,[2 4 6]), hold on,%figure of means incl confidence bounds
text (0.05, 0.95, 'd)', 'Units','normalized', 'FontSize', 12)
%for legend
k=1;i=1;
% Best & Bridge 1992
errorbar(mnshscBB, mnHhBB, sdHhBB, 'Color', colBB, 'LineWidth',1.5)
% Bennett1998
errorbar(mnshscBn, mnHhBn, sdHhBn, 'Color', colBn, 'LineWidth',1.5)
% Venditti2016
errorbar(mnshscVe(i), mnHhVe(i), sdHhVe(i),  'Color', colVe ,'LineWidth',1.5)
% Bradley2019
errorbar(cell2mat(dataBr{k}.Var10(2)), mnHhbr(k), sdHbr(k)./mnhbr(k), 'Color',colBr, 'LineWidth',1.5)
% de Lange & van de Vijsel (this paper)
errorbar(25, mean(dune_heights)/0.15, std(dune_heights/0.15), 'Color', colLV ,'LineWidth',1.5)
% % Naqshband
% errorbar(mnshscNa(k), mnHhNa(k), sdHhNa(k), 'c', 'LineWidth',1.5)

%bradley
for k= 1:length(sdHbr)
    errorbar(cell2mat(dataBr{k}.Var10(2)), mnHhbr(k), sdHhbr(k),sdHhbr(k), sdshscBr(k), sdshscBr(k), 'Color',colBr, 'LineWidth',1.5)
end

%venditti
for i = 1:3
errorbar(mnshscVe(i), mnHhVe(i), sdHhVe(i), sdHhVe(i), sdshscVe(i), sdshscVe(i), 'Color', colVe, 'LineWidth',1.5)
end

% Best & Bridge 1992
errorbar(mnshscBB, mnHhBB, sdHhBB, sdHhBB, sdshscBB, sdshscBB, 'Color',colBB, 'LineWidth',1.5)

% % Naqshband
% for k = 1:length(mnshscNa)
% errorbar(mnshscNa(k), mnHhNa(k), sdHhNa(k), 'c', 'LineWidth',1.5)
% end

plot(shsctst, Hhtst, 'k')
ylim ([0 0.6]); xlim ([0 40])
xlabel ('\theta / \theta_c [-]', 'FontSize', 12); ylabel ('\Delta / h [-]', 'FontSize', 12);
legend ('best1992', 'bennett1998', 'venditti2016', 'bradley2019', 'this paper', 'FontSize', 12, 'Location', 'southwest'); % 'this paper')
ax = gca; ax.FontSize = 12;

set(gcf,'Units','Inches');
pos = get(gcf,'Position');
set(gcf,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(gcf,['output/Figure1.pdf'],'-dpdf','-r500')
% print(gcf,['output/Figure1_rawdata_V2.png'],'-dpng','-r500')

%% processed data (Figure 3a and b)
close(figure(3));figure(3)
set(gcf,'Position',[50 50 1200 500])
subplot(1,2,1), hold on, %%figure with std
%for legend
k=1; i=1;
scatter(mnshscVe(i), sdHhVe(i), 80, 'MarkerFaceColor', colVe, 'MarkerEdgeColor', colVe)
scatter(cell2mat(dataBr{k}.Var10(2)), sdHhbr(k), 80, 'MarkerFaceColor', colBr, 'MarkerEdgeColor', colBr)
%add own data
scatter(25,  std(dune_heights/0.15), 80, 'MarkerFaceColor', colLV, 'MarkerEdgeColor', colLV)

%data venditti
for i = 1:3
    scatter(mnshscVe(i), sdHhVe(i), 80, 'MarkerFaceColor', colVe, 'MarkerEdgeColor', colVe)
end
%data bradley
for k = 1:length(dataBr)
scatter(cell2mat(dataBr{k}.Var10(2)), sdHhbr(k), 80, 'MarkerFaceColor', colBr, 'MarkerEdgeColor', colBr)
end

% combshsc = [mnshscBr'; mnshscLa(:); mnshscVe'];
% combhst = [sdHbr'; sdHLa(:); sdHVe'];
combshsc = [mnshscBr';  mnshscVe'; 25];
combhst = [sdHhbr';  sdHhVe'; std(dune_heights/0.15)];
md = fitlm(combshsc, combhst);
pl = plot(md);
delete(pl(1)); 
dataHandle = findobj(pl,'DisplayName','Data'); fitHandle = findobj(pl,'DisplayName','Fit');
cbHandles = findobj(pl,'DisplayName','Confidence bounds'); cbHandles = findobj(pl,'LineStyle',cbHandles.LineStyle, 'Color', cbHandles.Color);
fitHandle.Color = 'k'; set(cbHandles, 'Color', 'k', 'LineWidth', 1.5)

text(0.05, 0.9, strcat('R^2 = ', num2str(round(md.Rsquared.Ordinary, 2))), 'Units', 'normalized', 'FontSize',12)
text(0.05, 0.85, strcat('p = ', sprintf('%.2e',coefTest(md))), 'Units', 'normalized', 'FontSize',12)
text(0.05, 0.95, 'a)', 'Units', 'normalized', 'FontSize',12)



title([])
ylabel('\sigma (\Delta / h) [-]', 'FontSize',12,'Interpreter','tex'); xlabel ('\theta / \theta_c [-]', 'FontSize',12,'Interpreter','tex');
legend ('venditti2016', 'bradley2019', 'this paper', 'Location', 'southeast', 'FontSize',12)%, 'this paper')
ax = gca; ax.FontSize = 12;
xlim([4 30])
grid on

%%%% add fig with  modi
clear md pl
subplot (1,2,2), hold on,
scatter (sortmodi.T_gem, sortmodi.m1_m2, 80,sortmodi.m1_m2./CV','filled')
scatter(25, modi_exp(2) - modi_exp(1), 80, (modi_exp(2) - modi_exp(1))/CVLV, 'filled') %own experiment
cmap = flipud(cmocean('ice'));
colormap(cmap)
cb = colorbar;
ylabel(cb,'d_{modi} / CV [-]','Interpreter','tex')
cb.Location = 'south';
cb.FontSize = 12;
% ax = gca;  % Get current axis
% cb.Position = [0.5796    0.2365    0.3169    0.0427]; %[0.6, 0.1, 0.3, 0.05];
% scatter (sortmodi.T_gem(sig<0), sortmodi.m1_m2(sig<0),70,  'k')
% scatter (sortmodi.T_gem(sig>0), sortmodi.m1_m2(sig>0), 80,'filled', 'k')
% scatter(25, modi_exp(2) - modi_exp(1), 80, 'filled', 'k') %own experiment
md = fitlm ([modi.T_gem; 25], [modi.m1_m2; (modi_exp(2) - modi_exp(1))] );
pl = plot (md);
delete(pl(1)); 
dataHandle = findobj(pl,'DisplayName','Data'); fitHandle = findobj(pl,'DisplayName','Fit');
cbHandles = findobj(pl,'DisplayName','Confidence bounds'); cbHandles = findobj(pl,'LineStyle',cbHandles.LineStyle, 'Color', cbHandles.Color);
fitHandle.Color = 'k'; set(cbHandles, 'Color', 'k', 'LineWidth', 1.5)
legend off

text(0.05, 0.9, strcat('R^2 = ', num2str(round(md.Rsquared.Ordinary, 2))), 'Units', 'normalized', 'FontSize',12)
text(0.05, 0.85, strcat('p = ', sprintf('%.2e',coefTest(md))), 'Units', 'normalized', 'FontSize',12)
text(0.05, 0.95, 'b)', 'Units', 'normalized', 'FontSize',12)
title ([])
ylabel ('d_{modi} [-]', 'FontSize',12,'Interpreter','tex'); xlabel ('\theta / \theta_c [-]', 'FontSize',12,'Interpreter','tex');
% legend ('non-significant modi', 'significant modi', 'Location', 'southeast', 'FontSize',12)
ax = gca; ax.FontSize = 12;
xlim([4 30])
ylim([-0.05 0.30]-0.025)
grid on

set(gcf,'Units','Inches');
pos = get(gcf,'Position');
set(gcf,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(gcf,['output/Figure2.pdf'],'-dpdf','-r500')
% print(gcf,['Figure2_modi_V2.png'],'-dpng','-r500')

%% histogrammen (figure SI)

nrmeas = [60 39 75 30 24 31 28 30 30 28 28 30 26 29 28];
nrexp = 207;


close(figure(4));figure(4)
set(gcf,'Position', [50 50 1200 900]); 
for i = 1:length(x)-2
subplot(4,4,i)
nr = x(i);
histogram(table2array(dat(:,nr)), 'BinWidth', 0.03)
sortdat(:,i) = table2array(dat(:,nr));
xlim ([0 0.7])
title (strcat('T = ', num2str(round(sortallT(i),2))))
text(0.75, 0.9, strcat('n = ', num2str(nrmeas(i))), 'Units', 'Normalized')
xlabel ('\Delta/h [-]'); ylabel ('prohability')
xline (sortmodi.m1(i), 'r'); 
try
    xline (sortmodi.m2(i), 'r')
catch
end

end

%add own data
subplot(4,4,14)
histogram(dune_heights/0.15,  'BinWidth', 0.03)
xlim ([0 0.7])
title (strcat('T = ', num2str(25)))
xlabel ('\Delta/h [-]'); ylabel ('prohability')
text(0.75, 0.9, strcat('n = ', num2str(nrexp)), 'Units', 'Normalized')
xline (modi_exp, 'r'); 

for i = length(x):length(x)+1
subplot(4,4,i)
nr = x(i-1);
histogram(table2array(dat(:,nr)), 'BinWidth', 0.03)
sortdat(:,i-1) = table2array(dat(:,nr));
xlim ([0 0.7])
title (strcat('T = ', num2str(round(sortallT(i-1),2))))
text(0.75, 0.9, strcat('n = ', num2str(nrmeas(i-1))), 'Units', 'Normalized')
xlabel ('\Delta/h [-]'); ylabel ('prohability')
xline (sortmodi.m1(i-1), 'r'); 
try
    xline (sortmodi.m2(i-1), 'r')
catch
end

end

set(gcf,'Units','Inches');
pos = get(gcf,'Position');
set(gcf,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(gcf,['output/FigureS13.pdf'],'-dpdf','-r500')
% print(gcf,['SI_figure_14_histograms_V2.png'],'-dpng','-r500')


% %% exquation vR
% 
% % Ttst = [0:1:25]';
% theta = [0:0.1:5];
% thetac = 0.03;
% Tven = theta./thetac;
% Ttst = (theta.^2-thetac.^2)./thetac.^2;
% 
% htst = h.Var1;
% d50tst = 550*10^-6;
% for i = 1:15
% Hvrtst(:,i) = (0.11*htst(i) .*((d50tst./htst(i)).^0.3) .* (1-(exp(-0.5*Ttst'))).*(25-Ttst'))./htst(i);
% end
% 
% figure; hold on,
% plot(Tven, Hvrtst)
% plot (shsctst, Hhtst)
% ylim ([0 0.5])
% % xlabel ('(\theta^2 - \theta_c^2)/\theta'); 
% xlabel ('\theta / \theta_c')
% ylabel ('\Delta/h')


%% Plot mu, sigma, CV and d_modi

close(figure(10));figure(10);
% set(gcf,'Position',[50 50 550 1100])
fig=gcf;
fig.Units='normalized';
fig.OuterPosition=[0.10 0.10 0.30 0.5];
ms = 30;
colgrey = [0.8 0.8 0.8];
ind = ~isnan(sig); % indices of non-NaNs

tiledlayout(5,1, 'Padding', 'none', 'TileSpacing', 'compact'); 
% subplot(5,1,1); hold on
nexttile 
hold on
scatter (sortmodi.T_gem(ind), mn(ind)', ms,'k','filled')
scatter (sortmodi.T_gem(~ind), mn(~ind)', ms,'MarkerFaceColor', colgrey, 'MarkerEdgeColor', colgrey)
scatter(25, mnLV, ms,'k','filled') %own experiment
xline(18,'--')
% xlabel('T')
set(gca,'xticklabels',[])
ylabel('\mu')
xlim([0,30])
ylim([0 1.1*max(mn)])
grid on

% subplot(5,1,2); hold on
nexttile 
hold on
scatter (sortmodi.T_gem(ind),  stdd(ind)', ms,'k','filled')
scatter (sortmodi.T_gem(~ind), stdd(~ind)', ms,'MarkerFaceColor', colgrey, 'MarkerEdgeColor', colgrey)
scatter(25, stddLV, ms,'k','filled') %own experiment
xline(18,'--')
% xlabel('T')
set(gca,'xticklabels',[])
ylabel('\sigma')
xlim([0,30])
grid on

% subplot(5,1,3); hold on
nexttile 
hold on
scatter (sortmodi.T_gem(ind), CV(ind)', ms,'k','filled')
scatter (sortmodi.T_gem(~ind), CV(~ind)', ms,'MarkerFaceColor', colgrey, 'MarkerEdgeColor', colgrey)
scatter(25, CVLV, ms,'k','filled') %own experiment
xline(18,'--')
% xlabel('T')
set(gca,'xticklabels',[])
ylabel('CV = \sigma / \mu')
xlim([0,30])
grid on

% subplot(5,1,4); hold on
nexttile 
hold on
scatter (sortmodi.T_gem, sortmodi.m1_m2, ms,'k','filled')
scatter(25, modi_exp(2) - modi_exp(1), ms, 'k', 'filled') %own experiment
% cmap = flipud(cmocean('ice'));
% colormap(cmap)
% cb = colorbar;
% ylabel(cb,'d_{modi} / CV [-]','Interpreter','tex')
% cb.Location = 'north';
% cb.FontSize = 8;
xline(18,'--')
% xlabel('T')
set(gca,'xticklabels',[])
ylabel('d_{modi}')
xlim([0,30])
grid on

% subplot(5,1,5); hold on
nexttile 
hold on
scatter (sortmodi.T_gem, sortmodi.m1_m2./CV', ms,'k','filled')
scatter(25, (modi_exp(2) - modi_exp(1))/CVLV, ms,'k','filled') %own experiment
xline(18,'--')
yline(1,'--')
xlabel('\theta / \theta_c')
ylabel('d_{modi} / CV')
xlim([0,30])
grid on

% set(gcf,'Units','Inches');
% pos = get(gcf,'Position');
% set(gcf,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
% print(gcf,['CV_dmodi_sigma_mu.pdf'],'-dpdf','-r500')
% print(gcf,['CV_dmodi_sigma_mu.png'],'-dpng','-r500')
