%% DensityPlot_bedforms_V6.m

% This script creates a density plot (i.e. a "two-dimensional histogram")
% from time series (laboratory observations) of dune height as a function
% of transport stage. These observations are derived from the following two studies:
% - Venditti et al. (2016, Sedimentology) 
% - Bradley & Venditti (2019, JGR Earth Surface)
% In the current script, these raw data are shown in a two-dimensional
% density plot. Uni- and bimodal distributions have been fit to these data
% sets in another script. The outcomes of this previous analysis (i.e. the
% locations of the single or double modi, for each transport stage value)
% are here plotted in the density plot as well.
% Updates in Version 2: indicated when the bimodality is significant and
% fitted a quadratic relation through the modi.
% Updates in Version 3: Finetune the font size/type, add vertical dashed
% lines to locate the bullets, and calculate RMSE.
% Updates in Version 4: included the dune height values from our own
% experiment.
% Updates in Version 5: do not distinguish between significant or
% non-significant bimodality anymore: all bimodal distributions are significant.  
% Updates in Version 6: increase aspect ratio of the plot (make taller)

% Version 1 - 20 Nov 2023 - all OK.
% Version 2 - 01 Dec 2023 - all OK.
% Version 3 - 15 Dec 2023 - all OK.
% Version 4 - 07 Oct 2024 - all OK.
% Version 5 - 14 Feb 2025 - all OK.
% Version 6 - 11 Mar 2025 - all OK.

% Script written by: Roeland C. van de Vijsel
% Hydrology & Environmental Hydraulics Group, Wageningen University, the Netherlands

% This script belongs to the following publication:
% "Bimodality in subaqueous dune height suggests flickering behavior at high flow", published in Nature Communications (2025).
% Sjoukje I. de Lange(1,2,*), Roeland C. van de Vijsel(1), Paul J.J. F. Torfs(3), Nick P. Wallerstein(1), and A.J.F (Ton) Hoitink(1)
% 1 Wageningen University, Department of Environmental Sciences, Hydrology and Environmental Hydraulics Group, Wageningen, the Netherlands
% 2 HKV lijn in water, Lelystad, the Netherlands
% 3 independent researcher
% * Corresponding author: Sjoukje de Lange (delangesjoukje@gmail.com)

clear all
close all
clc

set(0, 'DefaultTextInterpreter', 'factory')
set(0, 'DefaultLegendInterpreter', 'factory')
set(0, 'DefaultAxesTickLabelInterpreter', 'factory')

on = 1; off = 0;
SAVE = on; % Whether or not figures should be saved

% Add the paths for input and output data
addpath input
addpath output

VERSION = 6;

% OK.

%% First load the data (dune height over average waterdepth, as function of transport stage)

% filename = 'input/data_av.xlsx';
filename = 'input/data_av_V4.xlsx'; % This is the updated input datasheet, where the values from our own experiment have been included

% Load the data
T = readtable(filename,'Sheet','duneheight over averagedepth');

% Row 0 (header) = transport stage, in words
% Row 1 = Transport stage
% Row 2 = Average water depth
% Row 3 - bottom row: duneheight over waterdepth (all values)

% Convert table to arrays (still unsorted in terms of the independent variable)

ts_unsorted = table2array(T(1,:));     % independent variable (here: transport stage)
wd_unsorted = table2array(T(2,:));     % independent variable (here: average waterdepth)
dh_unsorted = table2array(T(3:end,:)); % dependent variable (here: duneheight over average waterdepth)

clear T

% OK.

%% Second, load the modi

% filename = 'input/modi_distance.xlsx';
filename = 'input/modi_distance_V4.xlsx'; % This is the updated input datasheet, where the values from our own experiment have been included

% Load the data
T = readtable(filename,'Sheet','duneheight over average depth');

% Column 1 = run number
% Column 2 = transport stage
% Column 3 = value of mode 1
% Column 4 = value of mode 2 (if existing)
% Column 5 = distance between modes 1 and 2

% Convert table to arrays (still unsorted in terms of the independent variable)

ts2_unsorted = table2array(T(:,2))';    % independent variable (here: transport stage)
if all(ts_unsorted == ts2_unsorted)     % Check if this gives the same transport stages as found above
    clear ts2_unsorted
else
    error('inconsistent transport stages')
end

m1_unsorted = table2array(T(:,3))';  % dependent variable value of mode 1
m2_unsorted = table2array(T(:,4))';  % dependent variable value of mode 2 ( = NaN if non-existing, i.e. unimodal)

clear T filename

% OK.

%% Sort the dependent variable according to the independent variable

[ts,i_sort] = sort(ts_unsorted); % sorted transport stage, and sorting indices
wd = wd_unsorted(i_sort);   % average waterdepths, sorted according to transport stage
dh = dh_unsorted(:,i_sort); % duneheights over waterdepths, sorted according to transport stage
m1 = m1_unsorted(i_sort);   % values of mode 1, sorted according to transport stage
m2 = m2_unsorted(i_sort);   % values of mode 2, sorted according to transport stage

clear *_unsorted i_sort

% OK.

%% Sort the modi

modi = [m1;m2];
modi_sort = sort(modi,1);
modi_min = modi_sort(1,:); % the minimum of the two modi are all grouped together and always exist
modi_max = modi_sort(2,:); % the maximum of the modi is either existant or a NaN

% OK.

%% Convert to a string, in order to plot this along the x-axis later on

x_str = cell(1,length(ts)); % Make a string to store the values of the transport stage

for i = 1:length(ts)
    x_str{i} = sprintf('%.1f',ts(i));
end

clear i

% OK.

%% Determine the total range of (x,y) values

% values along the x-axis (here: transport stage)
xmin = min(ts(:));
xmax = max(ts(:));

% values along the y-axis (here: duneheight)
ymin = min(dh(:));
ymax = max(dh(:));

% OK.

%% Define the bins

nxbins = length(ts);
nybins = 20;

xbins = linspace(xmin,xmax,nxbins+1);
ybins = linspace(ymin,ymax,nybins+1);

% OK.

%% For each bin, determine the bin count

% Create arrays to store the bincounts
bincounts      = -999*ones(nxbins,nybins);
bincounts_norm = -999*ones(nxbins,nybins);

% Loop over all bins and count the number of datapoints per bin
for i = 1:nxbins
    
    % First select all y-data for this fixed value of x

    data = dh(:,i);
    data = data(~isnan(data)); % select only the non-NaN data
    N = length(data); % Total number of data points for this x-value

    for j = 1:nybins

        % Determine the bin edges in y-direction
        y0 = ybins(j);
        y1 = ybins(j+1);

        if (j < nybins)
        
            % Count the data points within this y-bin
            el = find( (data >= y0) & (data < y1) );
            bincounts(i,j) = length(el);
            bincounts_norm(i,j) = length(el)/N;

        elseif (j == nybins)

            % Count the data points within this y-bin
            el = find( (data >= y0) & (data <= y1) );
            bincounts(i,j) = length(el);
            bincounts_norm(i,j) = length(el)/N;

        end

    end
    
    disp(i)

end

% Check if the sum of bincounts adds up to the total number of datapoints

data_all = dh(:); % Flatten the 2D-array with all data
data_all = data_all(~isnan(data_all)); % Select only the non-NaN data
N_all = length(data_all); % Total number of data points

if(sum(bincounts(:)) == N_all)
    disp('OK: bincounts add up')
else
    error('bincounts do not add up')
end

% OK.

%% Theoretical dune height predictor (Bradley & Venditti 2019a, JGR Earth Surface)

ts_theory = linspace(min(ts),max(ts),100);
dh_theory = -0.001 * (ts_theory - 17.7).^2 + 0.417;

% OK.

%% Plot with equal bin widths
% Although the transport stage values (i.e., the values along the x-axis)
% are not equally spaced, we will first make the density plot while
% assuming that they are equidistant along the x-axis. Although this will
% "deform" the two-dimensional density plot in the x-direction, we will
% place XTickLabels with the exact transport stage value along the x-axis,
% to keep this transparent. At a later stage (hereafter), we will make a
% second density plot, where this non-equidistant x-spacing is visualized.

fs = 9;%7; %10; %12; % FontSize
lw = 1; %2;  % LineWidth

close(figure(1)); figure(1)

ymid = (ybins(1:end-1) + ybins(2:end))/2; % The midpoints of the y-bins

img = imagesc(ts,ymid,bincounts_norm');
set(gca,'YDir','Normal','FontSize',fs)

cmap = cmocean('ice'); % Choose a color scheme
colormap(flipud(cmap))
set(img, 'AlphaData', (1-(bincounts_norm' == 0))) % Make sure that bins whose prob. dens. is 0 become white

xlabel('transport stage [-]','FontSize',fs)
ylabel('dune height / water depth [-]','FontSize',fs)

x_str_loc = linspace(xmin,xmax,length(ts)); % The locations for the xticks (which are strings)
set(gca,'Xtick',x_str_loc)
set(gca,'XtickLabel',x_str)

cb = colorbar;
cb.Label.String = 'probability density [-]';
cb.Label.FontSize = fs; 

hold on

% Plot the modes

mc = [255,140,0]/255; % MarkerColor
ms = 6; %4; % MarkerSize

for ii = 1:length(ts)

    plot(x_str_loc(ii), modi_min(ii), 'o','MarkerEdgeColor',mc,'MarkerFaceColor',mc,'MarkerSize',ms)
    plot(x_str_loc(ii), modi_max(ii), 'o','MarkerEdgeColor','r','MarkerFaceColor','r','MarkerSize',ms)    

end

plot(ts_theory,dh_theory,'--','LineWidth',2,'Color',mc)

set(gca,'FontSize',fs)

axis([0,1.1*ceil(xmax),0,1.1*ceil(10*ymax)/10])
pbaspect([2,1,1])

xlabel('\theta / \theta_c [-]','FontSize',fs);
ylabel('\Delta / h [-]','FontSize',fs);

% Save as plot
% if SAVE == on
%     set(gcf,'Units','Inches');
%     pos = get(gcf,'Position');
%     set(gcf,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
%     print(gcf,['output/DensityPlot_bedforms_nybins=',num2str(nybins),'_equidistant.pdf'],'-dpdf','-r500')
%     print(gcf,['output/DensityPlot_bedforms_nybins=',num2str(nybins),'_equidistant.eps'],'-deps','-r500')
%     print(gcf,['output/DensityPlot_bedforms_nybins=',num2str(nybins),'_equidistant.png'],'-dpng','-r500')
% end

% OK.

%% Plot with unequal bin widths
% Now, instead of plotting each x-bin equally wide, we want to use the
% actual spacing between the measurements on the x-axis. That is, take the
% actual position of the transport stage values that have been used. Since
% these transport stage values are unequally spaced along the x-axis, we
% need to make the assumption that each x-bin ranges until halfway to the
% next transport stage value. For the smallest and largest transport stage
% value, this assumption cannot be made. Instead, for these outer
% boundaries, the width to the left (for minimal transport stage) and to
% the right (for maximal transport stage) is chosen equal to the width to
% the right and left, respectively.

close(figure(2));figure(2)

hold on

for ii = 1:length(ts) % Loop over all x-values (i.e. all transport stage values)

% For each selected x-value, plot a "vertical histogram":

    % Define the widths of each x-bin (halfway points between subsequent x-value)
    if (ii > 1) && (ii < length(ts))
        dts_pls = (ts(ii+1) - ts(ii)  )/2; % Half the distance to the next x-value
        dts_min = (ts(ii)   - ts(ii-1))/2; % Half the distance to the previous x-value
    elseif ii == 1
        dts_pls = (ts(ii+1) - ts(ii)  )/2; % Half the distance to the next x-value
        dts_min = dts_pls; % In absence of a previous x-value, choose the same spacing as to the next x-value
    elseif ii == length(ts)
        dts_min = (ts(ii)   - ts(ii-1))/2; % Half the distance to the previous x-value
        dts_pls = dts_min; % In absence of a next x-value, choose the same spacing as to the previous x-value
    end
    
    % Define the x- and y-edges of the bins to be plotted
    x_edges = ts(ii) + [-dts_min, dts_pls];
    y_edges = ybins;

    % Plot the "vertical histogram" for this value of transport stage, using colored rectangles as bins
    for j = 1:length(y_edges)-1
        
        % Specify the x- and y-edges of this bin
        x1 = x_edges(1);
        x2 = x_edges(2);
        y1 = y_edges(j);
        y2 = y_edges(j+1);
        
        % Calculate bin width and height
        width  = x2 - x1; 
        height = y2 - y1;
        
        bc_norm = bincounts_norm(ii,j);  % Normalized bin count value for this specific bin
        bc_min = min(bincounts_norm(:)); % Minimum value of all normalized bin counts
        bc_max = max(bincounts_norm(:)); % Maximum value of all normalized bin counts
        bc_norm_norm = (bc_norm - bc_min)/(bc_max - bc_min); % Normalize this specific normalized bincount again, now with respect to ALL the normalized bincount values

        % Map the value to a color (adjust colormap as needed)
        cmap = flipud(cmocean('ice')); % Choose a color scheme
        cmap(1,:) = [1,1,1]; % Make bins with value 0 white
        colormap(cmap)
        col_i = round(1 + bc_norm_norm*(size(cmap,1)-1)); % For each bincount value, select the corresponding index from the colormap
        color = cmap(col_i,:); % Specify the color that this bin should get
        rectangle('Position', [x1, y1, width, height], 'EdgeColor', 'none', 'FaceColor', color); % Draw the rectangle (bin)
    end

end

% Correct the label along the colorbar (use the "single" normalized bincount values, instead of the "double" normalized bincounts)
cbsteps = 6;% number of steps in which the colorbar label should be divided
cb_vec = linspace(bc_min,bc_max,cbsteps); % values to be plotted along the colorbar
for i = 1:cbsteps
    cb_str{i} = sprintf('%.2f',cb_vec(i)); % the labels to be plotted along the colorbar
end
cb = colorbar('XTick',linspace(0,1,cbsteps),'XTickLabel',cb_str,'FontSize',fs); % Place the colorbar label
cb.Label.String = 'probability density [-]';
cb.Label.FontSize = fs; 

set(gca,'XTick',ts,'XTickLabel',[])%,'XTickLabel',x_str) % Use x-label strings with less significant digits
xtickangle(45)

set(gca,'FontSize',fs)

axis([0,1.1*ceil(xmax),0,1.1*ceil(10*ymax)/10])
pbaspect([2,1,1])

ylabel('\Delta / h [-]','FontSize',fs);
% The xlabel needs to be placed further down, because it would otherwise overlap with the custommade XTickLabels (see below)
xlabel('\theta / \theta_c [-]','FontSize',fs,'Position',[mean([0,1.1*ceil(xmax)]), -0.125*ymax],'HorizontalAlignment','center');

% Plot the modes

mc = [255,140,0]/255; % MarkerColor
% ms = 4; % MarkerSize

for ii = 1:length(ts)

    plot(ts(ii), modi_min(ii), 'o','MarkerEdgeColor',mc,'MarkerFaceColor',mc,'MarkerSize',ms)
    plot(ts(ii), modi_max(ii), 'o','MarkerEdgeColor','r','MarkerFaceColor','r','MarkerSize',ms)    

end

plot(ts_theory,dh_theory,'--','LineWidth',2,'Color',mc) % Plot the theoretical relation


% The XTickLabels need to be slightly offset relative to their real
% position, because otherwise the labels of transport stage values that are
% too similar would overlap. The XTicks themselves will stay in the correct
% position, however.

texty = -0.04; % Vertical position of the new XTickLabels

for i = 1:length(ts)
    
    if (i == 6) || (i == 8) || (i==10)
        text(ts(i)-0.5,texty,x_str{i},'FontSize',fs,'HorizontalAlignment','center','Rotation',45)
    elseif (i == 7) || (i == 9)
        text(ts(i)+0.5,texty,x_str{i},'FontSize',fs,'HorizontalAlignment','center','Rotation',45)
    else
        text(ts(i),texty,x_str{i},'FontSize',fs,'HorizontalAlignment','center','Rotation',45)
    end

end

box on

% Save the plot
% if SAVE == on
%     set(gcf,'Units','Inches');
%     pos = get(gcf,'Position');
%     set(gcf,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
%     print(gcf,['output/DensityPlot_bedforms_nybins=',num2str(nybins),'_nonequidistant.pdf'],'-dpdf','-r500')
%     print(gcf,['output/DensityPlot_bedforms_nybins=',num2str(nybins),'_nonequidistant.eps'],'-deps','-r500')
%     print(gcf,['output/DensityPlot_bedforms_nybins=',num2str(nybins),'_nonequidistant.png'],'-dpng','-r500')
% end

% OK.

%% Plot with unequal bin widths; and fit quadratic curves to the modi
% Now, instead of plotting each x-bin equally wide, we want to use the
% actual spacing between the measurements on the x-axis. That is, take the
% actual position of the transport stage values that have been used. Since
% these transport stage values are unequally spaced along the x-axis, we
% need to make the assumption that each x-bin ranges until halfway to the
% next transport stage value. For the smallest and largest transport stage
% value, this assumption cannot be made. Instead, for these outer
% boundaries, the width to the left (for minimal transport stage) and to
% the right (for maximal transport stage) is chosen equal to the width to
% the right and left, respectively.

close(figure(3));figure(3)

hold on

for ii = 1:length(ts)
    xline(ts(ii),':','color',[0.5 0.5 0.5],'LineWidth',lw/2)
end

for ii = 1:length(ts) % Loop over all x-values (i.e. all transport stage values)

% For each selected x-value, plot a "vertical histogram":

    % Define the widths of each x-bin (halfway points between subsequent x-value)
    if (ii > 1) && (ii < length(ts))
        dts_pls = (ts(ii+1) - ts(ii)  )/2; % Half the distance to the next x-value
        dts_min = (ts(ii)   - ts(ii-1))/2; % Half the distance to the previous x-value
    elseif ii == 1
        dts_pls = (ts(ii+1) - ts(ii)  )/2; % Half the distance to the next x-value
        dts_min = dts_pls; % In absence of a previous x-value, choose the same spacing as to the next x-value
    elseif ii == length(ts)
        dts_min = (ts(ii)   - ts(ii-1))/2; % Half the distance to the previous x-value
        dts_pls = dts_min; % In absence of a next x-value, choose the same spacing as to the previous x-value
    end
    
    % Define the x- and y-edges of the bins to be plotted
    x_edges = ts(ii) + [-dts_min, dts_pls];
    y_edges = ybins;

    % Plot the "vertical histogram" for this value of transport stage, using colored rectangles as bins
    for j = 1:length(y_edges)-1
        
        % Specify the x- and y-edges of this bin
        x1 = x_edges(1);
        x2 = x_edges(2);
        y1 = y_edges(j);
        y2 = y_edges(j+1);
        
        % Calculate bin width and height
        width  = x2 - x1; 
        height = y2 - y1;
        
        bc_norm = bincounts_norm(ii,j);  % Normalized bin count value for this specific bin
        bc_min = min(bincounts_norm(:)); % Minimum value of all normalized bin counts
        bc_max = max(bincounts_norm(:)); % Maximum value of all normalized bin counts
        bc_norm_norm = (bc_norm - bc_min)/(bc_max - bc_min); % Normalize this specific normalized bincount again, now with respect to ALL the normalized bincount values

        % Map the value to a color (adjust colormap as needed)
        cmap = flipud(cmocean('ice')); % Choose a color scheme
        cmap(1,:) = [1,1,1]; % Make bins with value 0 white
        colormap(cmap)
        col_i = round(1 + bc_norm_norm*(size(cmap,1)-1)); % For each bincount value, select the corresponding index from the colormap
        color = cmap(col_i,:); % Specify the color that this bin should get
        rectangle('Position', [x1, y1, width, height], 'EdgeColor', 'none', 'FaceColor', color); % Draw the rectangle (bin)
    end

end

% Correct the label along the colorbar (use the "single" normalized bincount values, instead of the "double" normalized bincounts)
cbsteps = 6;% number of steps in which the colorbar label should be divided
cb_vec = linspace(bc_min,bc_max,cbsteps); % values to be plotted along the colorbar
for i = 1:cbsteps
    cb_str{i} = sprintf('%.2f',cb_vec(i)); % the labels to be plotted along the colorbar
end
cb = colorbar('XTick',linspace(0,1,cbsteps),'XTickLabel',cb_str,'FontSize',fs); % Place the colorbar label
cb.Label.String = 'probability density [-]';
cb.Label.FontSize = fs; 

set(gca,'XTick',ts,'XTickLabel',[])%,'XTickLabel',x_str) % Use x-label strings with less significant digits
xtickangle(45)

set(gca,'FontSize',fs)

axis([0,1.1*ceil(xmax),0,1.1*ceil(10*ymax)/10])
pbaspect([2,1,1])

ylabel('\Delta / h [-]','FontSize',fs);
% The xlabel needs to be placed further down, because it would otherwise overlap with the custommade XTickLabels (see below)
xlabel('\theta / \theta_c [-]','FontSize',fs,'Position',[mean([0,1.1*ceil(xmax)]), -0.13*ymax],'HorizontalAlignment','center');

% Plot the dune height predictor, fits through the modes, and the modes themselves

mc = [255,140,0]/255; % MarkerColor
% ms = 4; % MarkerSize

plot(ts_theory,dh_theory,'k-','LineWidth',lw) % Plot the theoretical relation

% Fit a quadratic relation to the minimal modi
xfit_min = ts(~isnan(modi_min)); % Select the non-NaN values
yfit_min = modi_min(~isnan(modi_min)); % Select the non-NaN values
[pmin,Smin] = polyfit(xfit_min,yfit_min,2); % Fit a quadratic function
xinterp_min = linspace(min(xfit_min),max(xfit_min),100); % Interpolate the x-values
yinterp_min = polyval(pmin,xinterp_min); % Find the corresponding values of the quadratic function

% Idem, but for the maximal modi
xfit_max = ts(~isnan(modi_max));
yfit_max = modi_max(~isnan(modi_max));
[pmax,Smax] = polyfit(xfit_max,yfit_max,2);
xinterp_max = linspace(min(xfit_max),max(xfit_max),100);
yinterp_max = polyval(pmax,xinterp_max);

% Calculate RMSE
A_min = yfit_min; % Actual values (minimal modi)
F_min = polyval(pmin,xfit_min); % Forecasted values (minimal modi)
E_min = rmse(F_min,A_min); % RMSE (minimal modi)
A_max = yfit_max; % Actual values (maximal modi)
F_max = polyval(pmax,xfit_max); % Forecasted values (maximal modi)
E_max = rmse(F_max,A_max); % RMSE (maximal modi)

% Plot the fitted quadratic functions
fit_min = plot(xinterp_min,yinterp_min,'--','Color',mc,'LineWidth',lw);
fit_max = plot(xinterp_max,yinterp_max,'--','Color','r','LineWidth',lw);

for ii = 1:length(ts)

    plot(ts(ii), modi_min(ii), 'o','MarkerEdgeColor',mc,'MarkerFaceColor',mc,'MarkerSize',ms,'LineWidth',lw/2)

    if ts(ii) < 18 % For transport stage < 18, the bimodality is not significant
        plot(ts(ii), modi_max(ii), 'o','MarkerEdgeColor','r','MarkerFaceColor','none','MarkerSize',ms,'LineWidth',lw/2)    
    elseif ts(ii) >= 18 % For transport stage > 18, the bimodality is significant
        plot(ts(ii), modi_max(ii), 'o','MarkerEdgeColor','r','MarkerFaceColor','r','MarkerSize',ms,'LineWidth',lw/2)    
    end

end

% The XTickLabels need to be slightly offset relative to their real
% position, because otherwise the labels of transport stage values that are
% too similar would overlap. The XTicks themselves will stay in the correct
% position, however.

texty = -0.06; % Vertical position of the new XTickLabels

for i = 1:length(ts)
    
    if (i == 6) || (i == 8) || (i==10)
        text(ts(i)-0.9,texty,x_str{i},'FontSize',fs,'HorizontalAlignment','center','Rotation',45)
    elseif (i == 7) || (i == 9)
        text(ts(i)+0.1,texty,x_str{i},'FontSize',fs,'HorizontalAlignment','center','Rotation',45)
    else
        text(ts(i),texty,x_str{i},'FontSize',fs,'HorizontalAlignment','center','Rotation',45)
    end

end

box on

legend([fit_max, fit_min],['RMSE = ',num2str(E_max,'%.3f')],['RMSE = ',num2str(E_min,'%.3f')],'Location','NorthWest','FontSize',fs)

% Save the plot
% if SAVE == on
%     set(gcf,'Units','Inches');
%     pos = get(gcf,'Position');
%     set(gcf,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
%     print(gcf,['output/DensityPlot_bedforms_nybins=',num2str(nybins),'_nonequidistant_V',num2str(VERSION),'.pdf'],'-dpdf','-r500')
%     print(gcf,['output/DensityPlot_bedforms_nybins=',num2str(nybins),'_nonequidistant_V',num2str(VERSION),'.eps'],'-deps','-r500')
%     print(gcf,['output/DensityPlot_bedforms_nybins=',num2str(nybins),'_nonequidistant_V',num2str(VERSION),'.png'],'-dpng','-r500')
% end

% OK.

%% Plot with unequal bin widths; and fit quadratic curves to the modi - UPDATE V5: make open bullets closed (no distinction non-significant/significant bimodality)
% Now, instead of plotting each x-bin equally wide, we want to use the
% actual spacing between the measurements on the x-axis. That is, take the
% actual position of the transport stage values that have been used. Since
% these transport stage values are unequally spaced along the x-axis, we
% need to make the assumption that each x-bin ranges until halfway to the
% next transport stage value. For the smallest and largest transport stage
% value, this assumption cannot be made. Instead, for these outer
% boundaries, the width to the left (for minimal transport stage) and to
% the right (for maximal transport stage) is chosen equal to the width to
% the right and left, respectively.

close(figure(3));figure(3)

hold on

for ii = 1:length(ts)
    xline(ts(ii),':','color',[0.5 0.5 0.5],'LineWidth',lw/2)
end

for ii = 1:length(ts) % Loop over all x-values (i.e. all transport stage values)

% For each selected x-value, plot a "vertical histogram":

    % Define the widths of each x-bin (halfway points between subsequent x-value)
    if (ii > 1) && (ii < length(ts))
        dts_pls = (ts(ii+1) - ts(ii)  )/2; % Half the distance to the next x-value
        dts_min = (ts(ii)   - ts(ii-1))/2; % Half the distance to the previous x-value
    elseif ii == 1
        dts_pls = (ts(ii+1) - ts(ii)  )/2; % Half the distance to the next x-value
        dts_min = dts_pls; % In absence of a previous x-value, choose the same spacing as to the next x-value
    elseif ii == length(ts)
        dts_min = (ts(ii)   - ts(ii-1))/2; % Half the distance to the previous x-value
        dts_pls = dts_min; % In absence of a next x-value, choose the same spacing as to the previous x-value
    end
    
    % Define the x- and y-edges of the bins to be plotted
    x_edges = ts(ii) + [-dts_min, dts_pls];
    y_edges = ybins;

    % Plot the "vertical histogram" for this value of transport stage, using colored rectangles as bins
    for j = 1:length(y_edges)-1
        
        % Specify the x- and y-edges of this bin
        x1 = x_edges(1);
        x2 = x_edges(2);
        y1 = y_edges(j);
        y2 = y_edges(j+1);
        
        % Calculate bin width and height
        width  = x2 - x1; 
        height = y2 - y1;
        
        bc_norm = bincounts_norm(ii,j);  % Normalized bin count value for this specific bin
        bc_min = min(bincounts_norm(:)); % Minimum value of all normalized bin counts
        bc_max = max(bincounts_norm(:)); % Maximum value of all normalized bin counts
        bc_norm_norm = (bc_norm - bc_min)/(bc_max - bc_min); % Normalize this specific normalized bincount again, now with respect to ALL the normalized bincount values

        % Map the value to a color (adjust colormap as needed)
        cmap = flipud(cmocean('ice')); % Choose a color scheme
        cmap(1,:) = [1,1,1]; % Make bins with value 0 white
        colormap(cmap)
        col_i = round(1 + bc_norm_norm*(size(cmap,1)-1)); % For each bincount value, select the corresponding index from the colormap
        color = cmap(col_i,:); % Specify the color that this bin should get
        rectangle('Position', [x1, y1, width, height], 'EdgeColor', 'none', 'FaceColor', color); % Draw the rectangle (bin)
    end

end

% Correct the label along the colorbar (use the "single" normalized bincount values, instead of the "double" normalized bincounts)
cbsteps = 6;% number of steps in which the colorbar label should be divided
cb_vec = linspace(bc_min,bc_max,cbsteps); % values to be plotted along the colorbar
for i = 1:cbsteps
    cb_str{i} = sprintf('%.2f',cb_vec(i)); % the labels to be plotted along the colorbar
end
cb = colorbar('XTick',linspace(0,1,cbsteps),'XTickLabel',cb_str,'FontSize',fs); % Place the colorbar label
cb.Label.String = 'probability density [-]';
cb.Label.FontSize = fs; 

set(gca,'XTick',ts,'XTickLabel',[])%,'XTickLabel',x_str) % Use x-label strings with less significant digits
xtickangle(45)

set(gca,'FontSize',fs)

axis([0,1.1*ceil(xmax),0,1.1*ceil(10*ymax)/10])
pbaspect([2,1,1])

ylabel('\Delta / h [-]','FontSize',fs);
% The xlabel needs to be placed further down, because it would otherwise overlap with the custommade XTickLabels (see below)
xlabel('\theta / \theta_c [-]','FontSize',fs,'Position',[mean([0,1.1*ceil(xmax)]), -0.13*ymax],'HorizontalAlignment','center');

% Plot the dune height predictor, fits through the modes, and the modes themselves

mc = [255,140,0]/255; % MarkerColor
% ms = 4; % MarkerSize

plot(ts_theory,dh_theory,'k-','LineWidth',lw) % Plot the theoretical relation

% Fit a quadratic relation to the minimal modi
xfit_min = ts(~isnan(modi_min)); % Select the non-NaN values
yfit_min = modi_min(~isnan(modi_min)); % Select the non-NaN values
[pmin,Smin] = polyfit(xfit_min,yfit_min,2); % Fit a quadratic function
xinterp_min = linspace(min(xfit_min),max(xfit_min),100); % Interpolate the x-values
yinterp_min = polyval(pmin,xinterp_min); % Find the corresponding values of the quadratic function

% Idem, but for the maximal modi
xfit_max = ts(~isnan(modi_max));
yfit_max = modi_max(~isnan(modi_max));
[pmax,Smax] = polyfit(xfit_max,yfit_max,2);
xinterp_max = linspace(min(xfit_max),max(xfit_max),100);
yinterp_max = polyval(pmax,xinterp_max);

% Calculate RMSE
A_min = yfit_min; % Actual values (minimal modi)
F_min = polyval(pmin,xfit_min); % Forecasted values (minimal modi)
E_min = rmse(F_min,A_min); % RMSE (minimal modi)
A_max = yfit_max; % Actual values (maximal modi)
F_max = polyval(pmax,xfit_max); % Forecasted values (maximal modi)
E_max = rmse(F_max,A_max); % RMSE (maximal modi)

% Plot the fitted quadratic functions
fit_min = plot(xinterp_min,yinterp_min,'--','Color',mc,'LineWidth',lw);
fit_max = plot(xinterp_max,yinterp_max,'--','Color','r','LineWidth',lw);

for ii = 1:length(ts)

    plot(ts(ii), modi_min(ii), 'o','MarkerEdgeColor',mc,'MarkerFaceColor',mc,'MarkerSize',ms,'LineWidth',lw/2)

    if ts(ii) < 18 % For transport stage < 18, the bimodality is not significant
        plot(ts(ii), modi_max(ii), 'o','MarkerEdgeColor','r','MarkerFaceColor','r','MarkerSize',ms,'LineWidth',lw/2)    
    elseif ts(ii) >= 18 % For transport stage > 18, the bimodality is significant
        plot(ts(ii), modi_max(ii), 'o','MarkerEdgeColor','r','MarkerFaceColor','r','MarkerSize',ms,'LineWidth',lw/2)    
    end

end

% The XTickLabels need to be slightly offset relative to their real
% position, because otherwise the labels of transport stage values that are
% too similar would overlap. The XTicks themselves will stay in the correct
% position, however.

texty = -0.06; % Vertical position of the new XTickLabels

for i = 1:length(ts)
    
    if (i == 6) || (i == 8) || (i==10)
        text(ts(i)-0.9,texty,x_str{i},'FontSize',fs,'HorizontalAlignment','center','Rotation',45)
    elseif (i == 7) || (i == 9)
        text(ts(i)+0.1,texty,x_str{i},'FontSize',fs,'HorizontalAlignment','center','Rotation',45)
    else
        text(ts(i),texty,x_str{i},'FontSize',fs,'HorizontalAlignment','center','Rotation',45)
    end

end

box on

legend([fit_max, fit_min],['RMSE = ',num2str(E_max,'%.3f')],['RMSE = ',num2str(E_min,'%.3f')],'Location','NorthWest','FontSize',fs)

% Save the plot
% if SAVE == on
%     set(gcf,'Units','Inches');
%     pos = get(gcf,'Position');
%     set(gcf,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
%     print(gcf,['output/DensityPlot_bedforms_nybins=',num2str(nybins),'_nonequidistant_V',num2str(VERSION),'.pdf'],'-dpdf','-r500')
%     print(gcf,['output/DensityPlot_bedforms_nybins=',num2str(nybins),'_nonequidistant_V',num2str(VERSION),'.eps'],'-deps','-r500')
%     print(gcf,['output/DensityPlot_bedforms_nybins=',num2str(nybins),'_nonequidistant_V',num2str(VERSION),'.png'],'-dpng','-r500')
% end

% OK.

%% Plot with unequal bin widths; and fit quadratic curves to the modi - UPDATE V5: make open bullets closed (no distinction non-significant/significant bimodality)
% Now, instead of plotting each x-bin equally wide, we want to use the
% actual spacing between the measurements on the x-axis. That is, take the
% actual position of the transport stage values that have been used. Since
% these transport stage values are unequally spaced along the x-axis, we
% need to make the assumption that each x-bin ranges until halfway to the
% next transport stage value. For the smallest and largest transport stage
% value, this assumption cannot be made. Instead, for these outer
% boundaries, the width to the left (for minimal transport stage) and to
% the right (for maximal transport stage) is chosen equal to the width to
% the right and left, respectively.

close(figure(3));figure(3)

hold on

for ii = 1:length(ts)
    xline(ts(ii),':','color',[0.5 0.5 0.5],'LineWidth',lw/2)
end

for ii = 1:length(ts) % Loop over all x-values (i.e. all transport stage values)

% For each selected x-value, plot a "vertical histogram":

    % Define the widths of each x-bin (halfway points between subsequent x-value)
    if (ii > 1) && (ii < length(ts))
        dts_pls = (ts(ii+1) - ts(ii)  )/2; % Half the distance to the next x-value
        dts_min = (ts(ii)   - ts(ii-1))/2; % Half the distance to the previous x-value
    elseif ii == 1
        dts_pls = (ts(ii+1) - ts(ii)  )/2; % Half the distance to the next x-value
        dts_min = dts_pls; % In absence of a previous x-value, choose the same spacing as to the next x-value
    elseif ii == length(ts)
        dts_min = (ts(ii)   - ts(ii-1))/2; % Half the distance to the previous x-value
        dts_pls = dts_min; % In absence of a next x-value, choose the same spacing as to the previous x-value
    end
    
    % Define the x- and y-edges of the bins to be plotted
    x_edges = ts(ii) + [-dts_min, dts_pls];
    y_edges = ybins;

    % Plot the "vertical histogram" for this value of transport stage, using colored rectangles as bins
    for j = 1:length(y_edges)-1
        
        % Specify the x- and y-edges of this bin
        x1 = x_edges(1);
        x2 = x_edges(2);
        y1 = y_edges(j);
        y2 = y_edges(j+1);
        
        % Calculate bin width and height
        width  = x2 - x1; 
        height = y2 - y1;
        
        bc_norm = bincounts_norm(ii,j);  % Normalized bin count value for this specific bin
        bc_min = min(bincounts_norm(:)); % Minimum value of all normalized bin counts
        bc_max = max(bincounts_norm(:)); % Maximum value of all normalized bin counts
        bc_norm_norm = (bc_norm - bc_min)/(bc_max - bc_min); % Normalize this specific normalized bincount again, now with respect to ALL the normalized bincount values

        % Map the value to a color (adjust colormap as needed)
        cmap = flipud(cmocean('ice')); % Choose a color scheme
        cmap(1,:) = [1,1,1]; % Make bins with value 0 white
        colormap(cmap)
        col_i = round(1 + bc_norm_norm*(size(cmap,1)-1)); % For each bincount value, select the corresponding index from the colormap
        color = cmap(col_i,:); % Specify the color that this bin should get
        rectangle('Position', [x1, y1, width, height], 'EdgeColor', 'none', 'FaceColor', color); % Draw the rectangle (bin)
    end

end

% Correct the label along the colorbar (use the "single" normalized bincount values, instead of the "double" normalized bincounts)
cbsteps = 6;% number of steps in which the colorbar label should be divided
cb_vec = linspace(bc_min,bc_max,cbsteps); % values to be plotted along the colorbar
for i = 1:cbsteps
    cb_str{i} = sprintf('%.2f',cb_vec(i)); % the labels to be plotted along the colorbar
end
cb = colorbar('XTick',linspace(0,1,cbsteps),'XTickLabel',cb_str,'FontSize',fs); % Place the colorbar label
cb.Label.String = 'probability density [-]';
cb.Label.FontSize = fs; 

set(gca,'XTick',ts,'XTickLabel',[])%,'XTickLabel',x_str) % Use x-label strings with less significant digits
xtickangle(45)

set(gca,'FontSize',fs)

axis([0,1.1*ceil(xmax),0,1.1*ceil(10*ymax)/10])
pbaspect([1.2,1,1])

ylabel('\Delta / h [-]','FontSize',fs);
% The xlabel needs to be placed further down, because it would otherwise overlap with the custommade XTickLabels (see below)
xlabel('\theta / \theta_c [-]','FontSize',fs,'Position',[mean([0,1.1*ceil(xmax)]), -0.13*ymax],'HorizontalAlignment','center');

% Plot the dune height predictor, fits through the modes, and the modes themselves

mc = [255,140,0]/255; % MarkerColor
% ms = 4; % MarkerSize

plot(ts_theory,dh_theory,'k-','LineWidth',lw) % Plot the theoretical relation

% Fit a quadratic relation to the minimal modi
xfit_min = ts(~isnan(modi_min)); % Select the non-NaN values
yfit_min = modi_min(~isnan(modi_min)); % Select the non-NaN values
[pmin,Smin] = polyfit(xfit_min,yfit_min,2); % Fit a quadratic function
xinterp_min = linspace(min(xfit_min),max(xfit_min),100); % Interpolate the x-values
yinterp_min = polyval(pmin,xinterp_min); % Find the corresponding values of the quadratic function

% Idem, but for the maximal modi
xfit_max = ts(~isnan(modi_max));
yfit_max = modi_max(~isnan(modi_max));
[pmax,Smax] = polyfit(xfit_max,yfit_max,2);
xinterp_max = linspace(min(xfit_max),max(xfit_max),100);
yinterp_max = polyval(pmax,xinterp_max);

% Calculate RMSE
A_min = yfit_min; % Actual values (minimal modi)
F_min = polyval(pmin,xfit_min); % Forecasted values (minimal modi)
E_min = rmse(F_min,A_min); % RMSE (minimal modi)
A_max = yfit_max; % Actual values (maximal modi)
F_max = polyval(pmax,xfit_max); % Forecasted values (maximal modi)
E_max = rmse(F_max,A_max); % RMSE (maximal modi)

% Plot the fitted quadratic functions
fit_min = plot(xinterp_min,yinterp_min,'--','Color',mc,'LineWidth',lw);
fit_max = plot(xinterp_max,yinterp_max,'--','Color','r','LineWidth',lw);

for ii = 1:length(ts)

    plot(ts(ii), modi_min(ii), 'o','MarkerEdgeColor',mc,'MarkerFaceColor',mc,'MarkerSize',ms,'LineWidth',lw/2)

    if ts(ii) < 18 % For transport stage < 18, the bimodality is not significant
        plot(ts(ii), modi_max(ii), 'o','MarkerEdgeColor','r','MarkerFaceColor','r','MarkerSize',ms,'LineWidth',lw/2)    
    elseif ts(ii) >= 18 % For transport stage > 18, the bimodality is significant
        plot(ts(ii), modi_max(ii), 'o','MarkerEdgeColor','r','MarkerFaceColor','r','MarkerSize',ms,'LineWidth',lw/2)    
    end

end

% The XTickLabels need to be slightly offset relative to their real
% position, because otherwise the labels of transport stage values that are
% too similar would overlap. The XTicks themselves will stay in the correct
% position, however.

texty = -0.06; % Vertical position of the new XTickLabels

for i = 1:length(ts)
    
    if (i == 6) || (i == 8) || (i==10)
        text(ts(i)-0.9,texty,x_str{i},'FontSize',fs,'HorizontalAlignment','center','Rotation',45)
    elseif (i == 7) || (i == 9)
        text(ts(i)+0.1,texty,x_str{i},'FontSize',fs,'HorizontalAlignment','center','Rotation',45)
    else
        text(ts(i),texty,x_str{i},'FontSize',fs,'HorizontalAlignment','center','Rotation',45)
    end

end

box on

legend([fit_max, fit_min],['RMSE = ',num2str(E_max,'%.3f')],['RMSE = ',num2str(E_min,'%.3f')],'Location','NorthWest','FontSize',fs)

% Save the plot
if SAVE == on
    set(gcf,'Units','Inches');
    pos = get(gcf,'Position');
    set(gcf,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
    % print(gcf,['output/DensityPlot_bedforms_nybins=',num2str(nybins),'_nonequidistant__taller_V',num2str(VERSION),'.pdf'],'-dpdf','-r500')
    % print(gcf,['output/DensityPlot_bedforms_nybins=',num2str(nybins),'_nonequidistant_taller_V',num2str(VERSION),'.eps'],'-deps','-r500')
    % print(gcf,['output/DensityPlot_bedforms_nybins=',num2str(nybins),'_nonequidistant_taller_V',num2str(VERSION),'.png'],'-dpng','-r500')
    print(gcf,['output/Figure3.pdf'],'-dpdf','-r500')
end

% OK.