%% analyse_data_ryanbradley.m

% This script analyses the raw bed scan data obtained in the flume
% experiments by Bradley & Venditti (2019).

% Version 1 - 09 Oct 2024 - All OK.

% Script written by: Roeland C. van de Vijsel
% Hydrology & Environmental Hydraulics Group, Wageningen University, the Netherlands

% This script belongs to the following publication:
% "Bimodality in subaqueous dune height suggests flickering behavior at high flow", published in Nature Communications (2025).
% Sjoukje I. de Lange(1,2,*), Roeland C. van de Vijsel(1), Paul J.J. F. Torfs(3), Nick P. Wallerstein(1), and A.J.F (Ton) Hoitink(1)
% 1 Wageningen University, Department of Environmental Sciences, Hydrology and Environmental Hydraulics Group, Wageningen, the Netherlands
% 2 HKV lijn in water, Lelystad, the Netherlands
% 3 independent researcher
% * Corresponding author: Sjoukje de Lange (delangesjoukje@gmail.com)

clear all
close all
clc

%% Setup parameters

seatek.N_sensors = 32; % Number of Seatek sensors used
sensor = 16; % Pick one sensor, for which the timeseries should be plotted

% OK.

%% Choose the run number for which the dataset should be imported

runNrs = [371,373,374,376,395,398,427,428,430,434,441,442,443,444,450,464];

j = 11; % Choose one run number
run = runNrs(j);
disp(['Selected run: ',num2str(run)])

% OK.

%% Prepare a list with all files to be imported
    
% Define the name and path of the "Raw Data" subfolder
if run == 371
    name1 = '15-BDLD (371)';
elseif run == 373
    name1 = '15-THLD (373)';    
elseif (run == 374) || (run == 376) 
    name1 = '20-THLD (374_376)';
    name1 = [name1,'/',num2str(run)];    
elseif run == 395
    name1 = '15-LMIX (395)';
elseif run == 398
    name1 = '15-UMIX (395)';
elseif (run == 427) || (run == 428) 
    name1 = '20-BDLD (427_428)';
    name1 = [name1,'/',num2str(run)];
elseif run == 430
    name1 = '25-THLD (430)';
elseif run == 434
    name1 = '20-SPSN(434)';
elseif (run == 441) || (run == 442) || (run == 443)
    name1 = '15-SPSN (441_442_443)';
    name1 = [name1,'/',num2str(run)];
elseif run == 444
    name1 = '20-UMIX (444)';
elseif run == 450
    name1 = '20-BDLD (450)';
elseif run == 464
    name1 = '20-LMIX (464)';
end

% Add the paths to the overarching folder containing raw and processed scans
name0 = '../Raw and Processed Scans'; 
name2 = 'Raw Data';
pathRaw = [name0,'/',name1,'/',name2,'/'];
pathProcessed = [name0,'/',name1,'/','Processed Data','/','Seatek Data'];
addpath([name0,'/',name1])
addpath(pathRaw)
addpath(pathProcessed)

% Select only the relevant files to be imported
pattern1 = [num2str(run),'_'];
pattern2 = '_Seatek_Scan.log';
pattern3 = '_Stepper_Pose.log';

% Obtain a file list with all relevant import files   
filePattern = fullfile(pathRaw, ['*',pattern2]);
fileList = dir(filePattern);
  
% Add a column to this file list with the scan number   
for i = 1:size(fileList,1)
    fileList(i).scanNr = str2double(strrep( strrep(fileList(i).name,pattern1,''), pattern2,''));
    disp(i)
end

% Sort the structure based on the "scanNr" values
[~, sortedIndices] = sort([fileList.scanNr]);

fileList = fileList(sortedIndices);

% OK.  

%% Load and plot all the scans

iskip = []; % Make an array to store the indices where no scan was made
ilist = 1:size(fileList,1); % The list of scans

% Prepare arrays that will store the min/max values and sizes of each scan
mins  = -999*size(fileList,1); % array to store min values
maxs  = -999*size(fileList,1); % array to store max values
q05s  = -999*size(fileList,1); % array to store 5% percentile values
q95s  = -999*size(fileList,1); % array to store 95% percentile values
sizes = -999*size(fileList,1); % array to store file sizes   

figure('units','normalized','outerposition',[0 0 1 1]) % Open a screen-sized figure

for i = ilist
        
    disp(i)
    
    sensorDataname = fileList(i).name; % The name of the Seatek sensor data file
    stepperDataname = strrep(sensorDataname,pattern2,pattern3); % The name of the corresponding stepper file
    
    sensorData = readtable([pathRaw,sensorDataname]); % Raw input data
    stepperData = readtable([pathRaw,stepperDataname]); % Raw input data
    
    % Skip input files that are empty
    if numel(sensorData)==0
        iskip = [iskip,i]; % Store this index, for later reference
        continue
    end
    
    % Obtain the horizontal position of each Seatek scan line over time
    stepperTime = stepperData.Var2; % Stepper cart: measurement times [milliseconds]
    stepperPos = stepperData.Var3;  % Stepper cart: corresponding horizontal positions [cm]
    sensorTime = sensorData.Var2;   % Seatek sensor: measurement times [milliseconds]
    sensorPos = interp1(stepperTime,stepperPos,sensorTime,'linear','extrap'); % Seatek sensor: corresponding horizontal positions [cm]
    
    % Add the Seatek sensor positions to the data table
    for ii = 1:size(sensorData,1)
    sensorData.Pos(ii) = sensorPos(ii);
    end
    
    % Load the data
    z_raw  = - table2array(sensorData(:,3:seatek.N_sensors+2)); % [cm] Sediment bed elevation (z; positive upwards) across the flume (x,y)
    x_raw  =   sensorData.Pos/100; % [m] Horizontal positions along the flume axis
    dy_raw = 2.5; % Cross-flume sensor spacing [cm]
    y_raw  = (0:dy_raw:(seatek.N_sensors-1)*dy_raw)/100'; % [m] Horizontal positions across the flume axis
    
    % Make a (x,y)-grid
    [x_raw_grid,y_raw_grid] = meshgrid(x_raw,y_raw); % Create a spatial [m] and temporal [min] grid, for plotting purposes
    x_raw_grid = x_raw_grid';
    y_raw_grid = y_raw_grid';
    
    % Plot each scan
    ax(i) = subplot(9,4,i);
    surf(x_raw_grid,y_raw_grid,z_raw,'EdgeColor','none'); view([0 90]) % Show the data, taking into account the non-equidistant spacing between sensors
    title(num2str(i))
    axis image

    % Calculate the min/max etc. of each scan
    mins(i) = nanmin(z_raw(:));
    maxs(i) = nanmax(z_raw(:));
    q05s(i) = quantile(z_raw(:),0.05);
    q95s(i) = quantile(z_raw(:),0.95);
    sizes(i) = size(z_raw,1);

end

% Calculate the min/max of all scans
minmins = min(mins);
maxmaxs = max(maxs);
minq05s = min(q05s);
maxq95s = max(q95s);
numsteps = min(sizes(sizes>0));

% Change the color range to the same range for each subplot
for i = ilist(ilist ~= iskip)
    caxis(ax(i), [minq05s maxq95s]); % Set the same caxis for each subplot
end

% OK.

%% Idem, but select 3 scans to plot

iSel = 33; % The selected index/scan

%%%%%%%%%%%%%%%% RAW DATA %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

sensorDataname = fileList(iSel).name; % The name of the Seatek sensor data file
stepperDataname = strrep(sensorDataname,pattern2,pattern3); % The name of the corresponding stepper file

sensorData = readtable([pathRaw,sensorDataname]); % Raw input data
stepperData = readtable([pathRaw,stepperDataname]); % Raw input data
       
% Obtain the horizontal position of each Seatek scan line over time
stepperTime = stepperData.Var2; % Stepper cart: measurement times [milliseconds]
stepperPos = stepperData.Var3;  % Stepper cart: corresponding horizontal positions [cm]
sensorTime = sensorData.Var2;   % Seatek sensor: measurement times [milliseconds]
sensorPos = interp1(stepperTime,stepperPos,sensorTime,'linear','extrap'); % Seatek sensor: corresponding horizontal positions [cm]
    
% Add the Seatek sensor positions to the data table
for ii = 1:size(sensorData,1)
sensorData.Pos(ii) = sensorPos(ii);
end
    
% Load the data
z_raw  = - table2array(sensorData(:,3:seatek.N_sensors+2)); % [cm] Sediment bed elevation (z; positive upwards) across the flume (x,y)
x_raw  =   sensorData.Pos/100; % [m] Horizontal positions along the flume axis
dy_raw = 2.5; % Cross-flume sensor spacing [cm]
y_raw  = (0:dy_raw:(seatek.N_sensors-1)*dy_raw)/100'; % [m] Horizontal positions across the flume axis
    
% Make a (x,y)-grid
[x_raw_grid,y_raw_grid] = meshgrid(x_raw,y_raw); % Create a spatial [m] and temporal [min] grid, for plotting purposes
x_raw_grid = x_raw_grid';
y_raw_grid = y_raw_grid';

%%%%%%%%%%%%%%%% FILTER THE RAW DATA (MOVING MEDIAN) %%%%%%%%%%%%%%%%%%%%%%

wx = 5; % Moving window width in the x-direction
wy = 5; % Moving window width in the x-direction

z_movmedianx = movmedian(z_raw,wx,1); % Only filter in x-direction
z_movmediany = movmedian(z_raw,wy,2); % Only filter in y-direction
z_movmedianxy = movmedian(z_movmedianx,wy,2); % Filter in x, then in y-dir
z_movmedianyx = movmedian(z_movmediany,wx,1); % Filter in y, then in x-dir

% OK.

%%%%%%%%%%%%%%% PLOT ALL MOVING MEDIAN OPTIONS %%%%%%%%%%%%%%%%%%%%%%%%%%%%

figure('units','normalized','outerposition',[0 0 1 1]) % Open a screen-sized figure

% Raw data
ax(1) = subplot(5,1,1);
surf(x_raw_grid,y_raw_grid,z_raw,'EdgeColor','none')
view([0 90])
title('raw')
axis image

ax(2) = subplot(5,1,2);
surf(x_raw_grid,y_raw_grid,z_movmedianx,'EdgeColor','none')
view([0 90])
title('mov x')
axis image

ax(3) = subplot(5,1,3);
surf(x_raw_grid,y_raw_grid,z_movmediany,'EdgeColor','none')
view([0 90])
title('mov y')
axis image

ax(4) = subplot(5,1,4);
surf(x_raw_grid,y_raw_grid,z_movmedianxy,'EdgeColor','none')
view([0 90])
title('mov xy')
axis image

ax(5) = subplot(5,1,5);
surf(x_raw_grid,y_raw_grid,z_movmedianyx,'EdgeColor','none')
view([0 90])
title('mov yx')
axis image

for i = 1:5
    caxis(ax(i), [minq05s maxq95s]); % Set the same caxis for each subplot
end

% OK.

%% Load the "clean" (processed by BV2019) data

seatek_clean = load([pathProcessed,'/','New_Seatek_XYZ_Clean_',num2str(run),'_',num2str(iSel),'.mat']);
seatek_clean = eval(['seatek_clean.New_Seatek_XYZ_Clean_',num2str(run),'_',num2str(iSel),';']);

Nsensors = 32;
Ncolumns = 4;
Nrows = size(seatek_clean,1);

% This data is still formatted as follows: 
% Columns 1, 5,  9, ...: x-coordinates (along-flume position) of sensors 1, 2, 3, ...
% Columns 2, 6, 10, ...: y-coordinates (cross-flume position) of sensors 1, 2, 3, ...
% Columns 3, 7, 11, ...: (?) offset, to be subtracted from z-coordinates for sensors 1, 2, 3, ...
% Columns 4, 8, 12, ...: z-coordinates (bed elevation) of sensors 1, 2, 3, ...
% Below, this data is reshaped such that column 1 contains all x, column 2
% contains all y, column 3 contains all offsets and column 4 contains all z

seatek_reshape = -999*ones(Nrows*Nsensors,Ncolumns); % Prepare a reshaped array

for i = 1:Nsensors

    % Range of indices in y-direction where the data should be taken from
    % in the original (unpractically structured) array
    ely1 = (i-1)*Ncolumns+1;
    ely2 = i*Ncolumns;

    % Range of indices in x-direction where data should be placed in the
    % reshaped array
    elx1 = (i-1)*Nrows+1;
    elx2 = i*Nrows;
    
    % Move the data from the original to the reshaped array
    seatek_reshape(elx1:elx2,:) = seatek_clean(:,ely1:ely2);

end

% Throw out Nans
el_nans = sum(isnan(seatek_reshape),2)>0; % Vector indicating which row contains an NaN
seatek_reshape(el_nans,:) = [];

% Define x, y and z vectors
x_processed = seatek_reshape(:,1)/100; % Along-flume coordinate [m]
y_processed = seatek_reshape(:,2)/100; % Cross-flume coordinate [m]
z_processed_vector = (seatek_reshape(:,4) - seatek_reshape(:,3)); % Bed elevation [cm]

Nrows2 = size(seatek_reshape,1)/Nsensors; % Number of remaining rows (after getting rid of NaNs)

% Grid the (x,y,z) data, assuming that the (x,y)-plane is uniformly gridded
xlin = linspace(min(x_processed), max(x_processed), Nrows2);
ylin = linspace(min(y_processed), max(y_processed), Nsensors);
[x_processed_grid,y_processed_grid] = meshgrid(xlin, ylin);
z_processed = griddata(x_processed,y_processed,z_processed_vector,x_processed_grid,y_processed_grid,'v4');

% Rotate to have the same dimensions as the raw data
x_processed_grid = x_processed_grid';
y_processed_grid = y_processed_grid';
z_processed = z_processed';

% OK.

%% Plot all filtering methods

lw = 1;
fs = 11;

z_processed_mean = mean(z_processed(:)); % Calculate the mean elevation, to subtract from the datasets

figure('units','normalized','outerposition',[0 0 1 1]) % Open a screen-sized figure

% Raw
ax1 = subplot(4,1,1); 
hold on
surf(x_processed_grid,y_processed_grid,z_raw(2:end,:)-z_processed_mean,'EdgeColor','none');
cmap = cmocean('delta','pivot',0); % Choose a color scheme
colormap(cmap);
yline(mean(y_processed_grid(:,sensor)),'k--','LineWidth',lw)
view([0 90])
axis tight;
% axis image
title('a) Raw')
cb = colorbar;
cb.FontSize = fs;
% yl = ylabel(cb,'Elevation relative to mean [cm]','Rotation',270,'FontSize',fs);
clim([min(z_processed(:)-z_processed_mean) max(z_processed(:)-z_processed_mean)])
set(gca,'FontSize',fs)

% Movmedian
ax2 = subplot(4,1,2);
hold on
surf(x_processed_grid,y_processed_grid,z_movmedianxy(2:end,:)-z_processed_mean,'EdgeColor','none');
cmap = cmocean('delta','pivot',0); % Choose a color scheme
colormap(cmap);
yline(mean(y_processed_grid(:,sensor)),'k--','LineWidth',lw)
view([0 90])
axis tight;
% axis image
title('b) Moving median')
cb = colorbar;
cb.FontSize = fs;
yl = ylabel(cb,'Elevation relative to mean [cm]','Rotation',270,'FontSize',fs);
ylabel('Cross-flume distance [m]')
clim([min(z_processed(:)-z_processed_mean) max(z_processed(:)-z_processed_mean)])
set(gca,'FontSize',fs)

ax3 = subplot(4,1,3);
hold on
surf(x_processed_grid,y_processed_grid,z_processed-z_processed_mean,'EdgeColor','none');
cmap = cmocean('delta','pivot',0); % Choose a color scheme
colormap(cmap);
yline(mean(y_processed_grid(:,sensor)),'k--','LineWidth',lw)
view([0 90])
axis tight;
% axis image
title('c) Processed')
cb = colorbar;
cb.FontSize = fs;
% yl = ylabel(cb,'Elevation relative to mean [cm]','Rotation',270,'FontSize',fs);
clim([min(z_processed(:)-z_processed_mean) max(z_processed(:)-z_processed_mean)])
set(gca,'FontSize',fs)

ax4 = subplot(4,1,4);
hold on
plot(x_processed_grid(:,sensor),z_raw(2:end,sensor)-z_processed_mean,'LineWidth',lw)
plot(x_processed_grid(:,sensor),z_movmedianxy(2:end,sensor)-z_processed_mean,'LineWidth',lw)
plot(x_processed_grid(:,sensor),z_processed(:,sensor)-z_processed_mean,'LineWidth',lw)
% colorbar
axis([min(x_processed_grid(:)) max(x_processed_grid(:)) min(z_processed(:)-z_processed_mean) max(z_processed(:)-z_processed_mean)])
legend('Raw','Moving median','Processed','Location','SouthWest','FontSize',fs)
xlabel('Along-flume distance [m]')
ylabel('Elevation relative to mean [cm]')
title('d) Cross-sections')
set(gca,'FontSize',fs)
clim([min(z_processed(:)-z_processed_mean) max(z_processed(:)-z_processed_mean)])
ax4.Position(3) = ax3.Position(3)-0.04;

dpi = 500;

% exportgraphics(gcf,['SI_Figure_XX_ryanbradley_filters_',num2str(run),'_',num2str(iSel),'.png'],'Resolution',dpi)
% exportgraphics(gcf,['SI_Figure_XX_ryanbradley_filters_',num2str(run),'_',num2str(iSel),'.pdf'],'Resolution',dpi) 
        % saveas(gcf,['SI_Figure_XX_ryanbradley_filters_',num2str(run),'_',num2str(iSel),'.fig']) 
exportgraphics(gcf,['output/FigureS11.pdf'],'Resolution',dpi) 

% OK.

%% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% PART II: MAKE AN (X,T)-DIAGRAM OF THE DATA IN RUN 443
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Make an (x,t) diagram
clear all
close all
clc

REDO_CALCULATIONS = 0; % Whether or not to redo the calculations to obtain an (x,t)-diagram. If not, previously saved output is loaded (to save time).


if REDO_CALCULATIONS == 1
    
    % Choose the run number for which the dataset should be imported
    
    pathBedXYZ = '../Raw and Processed Scans/15-SPSN (441_442_443)/443/BedXYZ';
    addpath(pathBedXYZ)
    
    Nsensors = 32;
    sensor = 16;
    
    ilist = [1:13,15:73,75,77:126];
    Nrows2_list = [];
    
    for i = ilist
    
        seatek_reshape{i} = importdata([pathBedXYZ,'/','xyz_443_',num2str(i),'.txt']);
        
        % Throw out Nans
        el_nans = sum(isnan(seatek_reshape{i}),2)>0; % Vector indicating which row contains an NaN
        seatek_reshape{i}(el_nans,:) = [];
        
        % Define x, y and z vectors
        x_processed = seatek_reshape{i}(:,1)/100; % Along-flume coordinate [m]
        y_processed = seatek_reshape{i}(:,2)/100; % Cross-flume coordinate [m]
        z_processed_vector = seatek_reshape{i}(:,3); % Bed elevation [cm]
        
        % Count the number of scan lines remaining (i.e. non-NaN)
        Nrows2 = size(seatek_reshape{i},1)/Nsensors; % Number of remaining rows (after getting rid of NaNs)
        Nrows2_list = [Nrows2_list, Nrows2]; % Number of scan lines in this bed scan
    
        % Grid the (x,y,z) data, assuming that the (x,y)-plane is uniformly gridded
        xlin = linspace(min(x_processed), max(x_processed), Nrows2);
        ylin = linspace(min(y_processed), max(y_processed), Nsensors);
        [x_processed_grid,y_processed_grid] = meshgrid(xlin, ylin);
        z_processed = griddata(x_processed,y_processed,z_processed_vector,x_processed_grid,y_processed_grid,'v4');
        
        % Rotate to have the same dimensions as the raw data
        x_processed_grid = x_processed_grid';
        y_processed_grid = y_processed_grid';
        z_processed = z_processed';
    
        X{i} = x_processed_grid;
        Y{i} = y_processed_grid;
        Z{i} = z_processed;
    
        disp(i)
    
    end
    
    save('output/Bradley_443_BedXYZ_output.mat','X','Y','Z')

    % OK.

elseif REDO_CALCULATIONS == 0

    % Load the previously saved output, so that this analysis does not need to be run again entirely
    load('output/Bradley_443_BedXYZ_output.mat')
    
    % OK. 

end

% OK.

%% Calculate the mean bed elevation of ALL scans together

ilist = [1:13,15:73,75,77:126];

means_arr = [];
for i = ilist
    Z_flatten = Z{i}(:);
    means_arr = [means_arr;Z_flatten];
end
meanmeanZ = nanmean(means_arr);
    
% OK.

%% Take the data from the middle sensor in each bed scan, to make an (x,t) diagram

Nscans = 126;
Nlines = 350; %max(Nrows2_list);
xt_diagram = NaN*ones(Nlines,Nscans);
sensor = 16;

% ilist = [1:13,15:73,75,77:126];

for i = ilist
    if size(X{i},1) == Nlines
        xt_diagram(:,i) = Z{i}(:,sensor) - meanmeanZ; %mean(Z{i}(:));
    elseif size(X{i},1) == Nlines-1
        xt_diagram(1:end-1,i) = Z{i}(:,sensor) - meanmeanZ; %mean(Z{i}(:));
    else
        error('different data size')
    end
end

xt_diagram = xt_diagram(:,ilist); % Discard rows with no scan data
xt_diagram = xt_diagram(1:end-1,:);

scan_times = readtable('443_scan_times.xlsx');
scan_numbers = table2array(scan_times(2,2:end));
scan_times =  60*60*table2array(scan_times(3,2:end)); % Scan times [s]
scan_times = scan_times - scan_times(1); % Let the time vector start at 0

xt_xvec = X{1}(1:end-1,1);
xt_tvec = scan_times;

xt_xarr = repmat(xt_xvec, 1, numel(xt_tvec));
xt_tarr = repmat(xt_tvec, numel(xt_xvec), 1);

% OK.

%% Plot the (x,t)-diagram, alongside a few examples of full bed scans and a "cross-section" trough the (x,t)-diagram (i.e., a timeseries)

figure('units','normalized','outerposition',[0 0 1 1]) % Open a screen-sized figure

fs = 11;
lw = 1;

% Select for which times a full bed scan should be shown (show a few scans with high dunes and a few with flattened bed)
el_time1 = 107; % flat
el_time2 = 79; % high dunes
el_time3 = 46; % flat
el_time4 = 20; % high dunes

% Make a copy of the (x,t)-array, where the rows above and below the
% selected rows are made into a NaN, to highlight the selected rows.
xt_diagram2 = xt_diagram;
xt_diagram2(:,[ find(ilist == el_time1)-1,find(ilist == el_time1)+1,...
                find(ilist == el_time2)-1,find(ilist == el_time2)+1,...
                find(ilist == el_time3)-1,find(ilist == el_time3)+1,...
                find(ilist == el_time4)-1,find(ilist == el_time4)+1,]) = NaN;

% Select which position in the along-flume distance should be taken to compute a timeseries
el_select = round(Nlines/2); % Take the one halfway the length (x-axis) of the flume
timeseries = xt_diagram(el_select,:);

% Plot the timeseries
ax(1)=subplot(4,2,[1:2:8]);
hold on
imagesc(xt_xvec,xt_tvec/60,xt_diagram2'); set(gca,'YDir','Normal')
view([0 90])
axis([min(xt_xvec) max(xt_xvec) min(xt_tvec)/60 max(xt_tvec)/60])
axis tight;
xlabel('Along-flume distance [m]','FontSize',fs)
ylabel('Flume run time [min]','FontSize',fs)
cb = colorbar;
cb.FontSize = fs;
yl = ylabel(cb,'Bed elevation relative to mean [cm]','Rotation',270,'FontSize',fs);
cb.Position(1) = cb.Position(1)+0.50;
clim([min(xt_diagram(:)) max(xt_diagram(:))])
text(1.05*max(xt_xvec),scan_times(find(ilist == el_time1))/60,'$\longrightarrow$','Interpreter','latex')
text(1.05*max(xt_xvec),scan_times(find(ilist == el_time2))/60,'$\longrightarrow$','Interpreter','latex')
text(1.05*max(xt_xvec),scan_times(find(ilist == el_time3))/60,'$\longrightarrow$','Interpreter','latex')
text(1.05*max(xt_xvec),scan_times(find(ilist == el_time4))/60,'$\longrightarrow$','Interpreter','latex')
text(0.05,0.98,'a)','Units','Normalized','FontSize',fs,'FontWeight','bold')
cmap = cmocean('delta','pivot',0); % Choose a color scheme
colormap(cmap)


% Plot a full bed scan
ax(2)=subplot(4,2,[2]);
% surf(X{el_time1},Y{el_time1},Z{el_time1} - mean(Z{el_time1}(:)),'EdgeColor','none');
surf(X{el_time1},Y{el_time1},Z{el_time1} - meanmeanZ,'EdgeColor','none');
clim([min(xt_diagram(:)) max(xt_diagram(:))])
yline(mean(Y{el_time1}(:,sensor)),'k:','LineWidth',lw+1)
% colorbar
view([0 90])
axis([min(X{el_time1}(:)) max(X{el_time1}(:)) min(Y{el_time1}(:)) max(Y{el_time1}(:))])
axis tight
text(0.01,0.90,'b)','Units','Normalized','FontSize',fs,'FontWeight','bold')

% Plot a full bed scan
ax(3)=subplot(4,2,[4]);
% surf(X{el_time2},Y{el_time2},Z{el_time2} - mean(Z{el_time2}(:)),'EdgeColor','none');
surf(X{el_time2},Y{el_time2},Z{el_time2} - meanmeanZ,'EdgeColor','none');
clim([min(xt_diagram(:)) max(xt_diagram(:))])
yline(mean(Y{el_time2}(:,sensor)),'k:','LineWidth',lw+1)
view([0 90])
axis([min(X{el_time2}(:)) max(X{el_time2}(:)) min(Y{el_time2}(:)) max(Y{el_time2}(:))])
axis tight
text(0.01,0.90,'c)','Units','Normalized','FontSize',fs,'FontWeight','bold')

% Plot a full bed scan
ax(4)=subplot(4,2,[6]);
% surf(X{el_time3},Y{el_time3},Z{el_time3} - mean(Z{el_time3}(:)),'EdgeColor','none');
surf(X{el_time3},Y{el_time3},Z{el_time3} - meanmeanZ,'EdgeColor','none');
clim([min(xt_diagram(:)) max(xt_diagram(:))])
yline(mean(Y{el_time3}(:,sensor)),'k:','LineWidth',lw+1)
view([0 90])
axis([min(X{el_time3}(:)) max(X{el_time3}(:)) min(Y{el_time3}(:)) max(Y{el_time3}(:))])
axis tight
ylabel('                                           Cross-flume distance [m]')
text(0.01,0.90,'d)','Units','Normalized','FontSize',fs,'FontWeight','bold')

% Plot a full bed scan
ax(5)=subplot(4,2,[8]);
% surf(X{el_time4},Y{el_time4},Z{el_time4} - mean(Z{el_time4}(:)),'EdgeColor','none');
surf(X{el_time4},Y{el_time4},Z{el_time4} - meanmeanZ,'EdgeColor','none');
clim([min(xt_diagram(:)) max(xt_diagram(:))])
yline(mean(Y{el_time4}(:,sensor)),'k:','LineWidth',lw+1)
view([0 90])
axis([min(X{el_time4}(:)) max(X{el_time4}(:)) min(Y{el_time4}(:)) max(Y{el_time4}(:))])
axis tight
xlabel('Along-flume distance [m]')
text(0.01,0.90,'e)','Units','Normalized','FontSize',fs,'FontWeight','bold')

set(ax,'FontSize',fs)

% Save the figure
dpi = 500;
% exportgraphics(gcf,['SI_Figure_XX_ryanbradley_xt_run443.png'],'Resolution',dpi)
% exportgraphics(gcf,['SI_Figure_XX_ryanbradley_xt_run443.pdf'],'Resolution',dpi) 
        % saveas(gcf,['SI_Figure_XX_ryanbradley_xt_run443.fig']) 
exportgraphics(gcf,['output/FigureS12.pdf'],'Resolution',dpi) 

% OK.

%% Calculate the long-term development of mean bed level

meanbedlevels = -999*ones(size(ilist));
ii = 0;
for i = ilist
ii = ii + 1;
meanbedlevels(ii) = mean(Z{i}(:));
end

figure
plot(scan_times/60,meanbedlevels-meanmeanZ)
yline(0)
set(gca,'XLim',[min(scan_times/60) max(scan_times/60)])
xlabel('Flume run time [min]')
ylabel('Scan-averaged bed level minus overall mean [cm]')
pbaspect([2 1 1])
% Save the figure
% dpi = 500;
% exportgraphics(gcf,['Ryanbradley_meanbedlevels_run443.png'],'Resolution',dpi)
% exportgraphics(gcf,['Ryanbradley_meanbedlevels_run443.pdf'],'Resolution',dpi) 
%         saveas(gcf,['Ryanbradley_meanbedlevels_run443.fig']) 


%%
figure('units','normalized','outerposition',[0 0 0.5 1]) % Open a screen-sized figure
% Sample data creation (replace this with your actual data)
NY = 25%14; % number of rows
NX = 5%9; % number of columns
fs = 10; % Font size

ii = 0;
for i = ilist
    ii=ii+1;
    % Create subplot
    ax(ii) = subplot(NY, NX, ii); 
    
    % Create surf plot
    surf(X{i}, Y{i}, Z{i} - meanmeanZ, 'EdgeColor', 'none'); 
    caxis([min(xt_diagram(:)) max(xt_diagram(:))]); % Set color limits
    
    % Set view and axis limits
    view([0 90]);
    axis([min(X{i}(:)) max(X{i}(:)) min(Y{i}(:)) max(Y{i}(:))]); 
    pbaspect([max(X{i}(:))-min(X{i}(:)) max(Y{i}(:))-min(Y{i}(:)) 1])
    axis tight;

    % Remove ticks
    set(ax(ii), 'XTick', [], 'YTick', []);
    cmap = cmocean('delta','pivot',0); % Choose a color scheme
    colormap(cmap)
    % colorbar

    % Add subplot index text
    text(-0.01, 0.50, [num2str(scan_times(ii)/60/60,'%.1f')], 'Units', 'Normalized', 'FontSize', fs, 'FontWeight', 'bold','HorizontalAlignment','right','VerticalAlignment','middle');
end

% Adjust subplot positions for tight spacing
% for i = 1:length(ax)
%     pos = get(ax(i), 'Position');
%     pos(1) = pos(1) - 0.001; % adjust x position
%     pos(2) = pos(2) - 0.001; % adjust y position
%     pos(3) = pos(3) - 0.002; % adjust width
%     pos(4) = pos(4) - 0.002; % adjust height
%     set(ax(i), 'Position', pos);
% end

% dpi = 500;
% exportgraphics(gcf,['Bradley_run443_allscans.png'],'Resolution',dpi)
% exportgraphics(gcf,['Bradley_run443_allscans.pdf'],'Resolution',dpi) 
%         saveas(gcf,['Bradley_run443_allscans.fig']) 
% Make the figure fit on the screen
% set(gcf, 'Position', [100, 100, 1200, 800]); % Change size as necessary

%% OLD: Plot the (x,t)-diagram, alongside a few examples of full bed scans and a "cross-section" trough the (x,t)-diagram (i.e., a timeseries)
% 
% figure('units','normalized','outerposition',[0 0 1 1]) % Open a screen-sized figure
% 
% fs = 11;
% lw = 1;
% 
% % Select for which times a full bed scan should be shown
% el_time1 = 107%111;
% el_time2 = 79%78;
% el_time3 = 46%45;
% el_time4 = 20%12;
% 
% % Select which position in the along-flume distance should be taken to compute a timeseries
% el_select = round(Nlines/2); % Take the one halfway the length (x-axis) of the flume
% timeseries = xt_diagram(el_select,:);
% 
% % Plot the timeseries
% ax(1)=subplot(4,5,[1:5:20]);
% hold on
% xline(0)
% plot(timeseries,xt_tvec/60,'b','LineWidth',lw)
% yline(scan_times(el_time1)/60,'k:','LineWidth',lw+1)
% yline(scan_times(el_time2)/60,'k:','LineWidth',lw+1)
% yline(scan_times(el_time3)/60,'k:','LineWidth',lw+1)
% yline(scan_times(el_time4)/60,'k:','LineWidth',lw+1)
% axis([min(timeseries) max(timeseries) min(xt_tvec)/60 max(xt_tvec/60)])
% ylabel('Flume run time [min]','FontSize',fs)
% xlabel('Elevation relative to mean [cm]')
% text(0.05,0.98,'a)','Units','Normalized','FontSize',fs,'FontWeight','bold')
% 
% % Plot the (x,t)-diagram
% ax(2)=subplot(4,5,[2:5:20]);
% hold on
% surf(xt_xarr,xt_tarr/60,xt_diagram,'EdgeColor','none');
% cmap = cmocean('delta','pivot',0); % Choose a color scheme
% colormap(cmap);
% yline(scan_times(el_time1)/60,'k:','LineWidth',lw+1)
% yline(scan_times(el_time2)/60,'k:','LineWidth',lw+1)
% yline(scan_times(el_time3)/60,'k:','LineWidth',lw+1)
% yline(scan_times(el_time4)/60,'k:','LineWidth',lw+1)
% xline(xt_xvec(el_select),'b--','LineWidth',lw+1)
% view([0 90])
% axis([min(xt_xvec) max(xt_xvec) min(xt_tvec)/60 max(xt_tvec)/60])
% % axis tight;
% xlabel('Along-flume distance [m]','FontSize',fs)
% cb = colorbar;
% cb.FontSize = fs;
% yl = ylabel(cb,'Elevation relative to mean [cm]','Rotation',270,'FontSize',fs);
% cb.Position(1) = cb.Position(1)+0.52;
% clim([min(xt_diagram(:)) max(xt_diagram(:))])
% text(1.03*max(xt_xvec),scan_times(el_time1)/60,'$\longrightarrow$','Interpreter','latex')
% text(1.03*max(xt_xvec),scan_times(el_time2)/60,'$\longrightarrow$','Interpreter','latex')
% text(1.03*max(xt_xvec),scan_times(el_time3)/60,'$\longrightarrow$','Interpreter','latex')
% text(1.03*max(xt_xvec),scan_times(el_time4)/60,'$\longrightarrow$','Interpreter','latex')
% text(0.05,0.98,'b)','Units','Normalized','FontSize',fs,'FontWeight','bold')
% 
% % Plot a full bed scan
% ax(3)=subplot(4,5,[3:5]);
% surf(X{el_time1},Y{el_time1},Z{el_time1} - mean(Z{el_time1}(:)),'EdgeColor','none');
% clim([min(xt_diagram(:)) max(xt_diagram(:))])
% yline(mean(Y{el_time1}(:,sensor)),'k:','LineWidth',lw+1)
% % colorbar
% view([0 90])
% axis([min(X{el_time1}(:)) max(X{el_time1}(:)) min(Y{el_time1}(:)) max(Y{el_time1}(:))])
% axis image
% text(0.01,0.90,'f)','Units','Normalized','FontSize',fs,'FontWeight','bold')
% 
% % Plot a full bed scan
% ax(4)=subplot(4,5,[8:10]);
% surf(X{el_time2},Y{el_time2},Z{el_time2} - mean(Z{el_time2}(:)),'EdgeColor','none');
% clim([min(xt_diagram(:)) max(xt_diagram(:))])
% yline(mean(Y{el_time2}(:,sensor)),'k:','LineWidth',lw+1)
% view([0 90])
% axis([min(X{el_time2}(:)) max(X{el_time2}(:)) min(Y{el_time2}(:)) max(Y{el_time2}(:))])
% axis image
% text(0.01,0.90,'e)','Units','Normalized','FontSize',fs,'FontWeight','bold')
% 
% % Plot a full bed scan
% ax(5)=subplot(4,5,[13:15]);
% surf(X{el_time3},Y{el_time3},Z{el_time3} - mean(Z{el_time3}(:)),'EdgeColor','none');
% clim([min(xt_diagram(:)) max(xt_diagram(:))])
% yline(mean(Y{el_time3}(:,sensor)),'k:','LineWidth',lw+1)
% view([0 90])
% axis([min(X{el_time3}(:)) max(X{el_time3}(:)) min(Y{el_time3}(:)) max(Y{el_time3}(:))])
% axis image
% ylabel('                                           Cross-flume distance [m]')
% text(0.01,0.90,'d)','Units','Normalized','FontSize',fs,'FontWeight','bold')
% 
% % Plot a full bed scan
% ax(6)=subplot(4,5,[18:20]);
% surf(X{el_time4},Y{el_time4},Z{el_time4} - mean(Z{el_time4}(:)),'EdgeColor','none');
% clim([min(xt_diagram(:)) max(xt_diagram(:))])
% yline(mean(Y{el_time4}(:,sensor)),'k:','LineWidth',lw+1)
% view([0 90])
% axis([min(X{el_time4}(:)) max(X{el_time4}(:)) min(Y{el_time4}(:)) max(Y{el_time4}(:))])
% axis image
% xlabel('Along-flume distance [m]')
% text(0.01,0.90,'c)','Units','Normalized','FontSize',fs,'FontWeight','bold')
% 
% set(ax,'FontSize',fs)
% 
% % Save the figure
% dpi = 500;
% exportgraphics(gcf,['SI_Figure_XX_ryanbradley_xt_run443.png'],'Resolution',dpi)
% exportgraphics(gcf,['SI_Figure_XX_ryanbradley_xt_run443.pdf'],'Resolution',dpi) 
%         saveas(gcf,['SI_Figure_XX_ryanbradley_xt_run443.fig']) 
% 
% % OK.