import requests
import json
import argparse
import time
import warnings

def fetch_quantum_random_hex(num_bytes=4, verify_ssl=False):
    """
    Fetches quantum random hexadecimal numbers from the BYU Camacho Lab QRNG API.
    
    Args:
        num_bytes (int): Number of bytes to generate (default: 4)
        verify_ssl (bool): Whether to verify SSL certificates (default: False)
        
    Returns:
        str: Hexadecimal string
    """
    url = f"https://camacholab.ee.byu.edu/qrng/hex/{num_bytes}"
    
    if not verify_ssl:
        warnings.warn("SSL certificate verification is disabled. This is potentially insecure.")
    
    try:
        response = requests.get(url, verify=verify_ssl)
        response.raise_for_status()
        # The response directly contains the hex string
        return response.text.strip()
    except requests.exceptions.RequestException as e:
        print(f"Error fetching data: {e}")
        return None

def generate_and_save_hex_numbers(n, output_file='quantum_random_hex.json', verify_ssl=False):
    """
    Generates n 4-byte hexadecimal numbers and saves them to a JSON file.
    
    Args:
        n (int): Number of hexadecimal numbers to generate
        output_file (str): Name of the output JSON file
        verify_ssl (bool): Whether to verify SSL certificates
    """
    hex_numbers = []
    
    print(f"Generating {n} quantum random hexadecimal numbers...")
    
    for i in range(n):
        hex_value = fetch_quantum_random_hex(4, verify_ssl)  # 4 bytes = 8 hex characters
        if hex_value:
            hex_numbers.append(hex_value)
            print(f"Generated {i+1}/{n}: {hex_value}")
        else:
            print(f"Failed to generate number {i+1}/{n}")
        
        # Add a small delay to avoid overwhelming the server
        # if i < n-1:  # Don't delay after the last request
            # time.sleep(0.5)
    
    # Save the numbers to a JSON file
    with open(output_file, 'w') as f:
        json.dump({"quantum_random_hex": hex_numbers}, f, indent=2)
    
    print(f"Successfully saved {len(hex_numbers)} numbers to {output_file}")

def main():
    # Set up command line argument parsing
    parser = argparse.ArgumentParser(description='Generate quantum random hexadecimal numbers')
    parser.add_argument('n', type=int, help='Number of hexadecimal numbers to generate')
    parser.add_argument('--output', '-o', default='quantum_random_hex.json', 
                        help='Output JSON file name (default: quantum_random_hex.json)')
    parser.add_argument('--verify-ssl', action='store_true', 
                        help='Verify SSL certificates (default: False)')
    
    args = parser.parse_args()
    
    if args.n <= 0:
        print("Error: Number of numbers to generate must be positive")
        return
    
    generate_and_save_hex_numbers(args.n, args.output, args.verify_ssl)

if __name__ == "__main__":
    main()