function [ad1T, ad2U] = ...
        TestBoxImplicit(ad2Mass, WidthLoad, Fmin, Fmax, Freq, Ncyc, Ti, dt, MODAL, PLOT, VideoName)
if nargin < 7, Ti = 1; end
if nargin < 8, dt = 0.001; end
if nargin < 9, MODAL = 0; end
if nargin < 10, PLOT = 0; end
if nargin < 11, VideoName = []; end
Ti = -Ti;

%% Calculate grains in contact with load
LoadedGrain = [];
for iMass = 1: 1: size(ad2Mass,1)
    if ad2Mass(iMass,2) == 0 && abs(ad2Mass(iMass,1)) <= WidthLoad/2
        LoadedGrain = [LoadedGrain; iMass];
    end
end

%% Solve ode
[ad1T, ad2U, ad2V, ad2A] = Implicit_NL(@(t,ad1U, ad1V)ForcesAndMatrices(t,ad1U,ad1V,...
                    ad2Mass,LoadedGrain,Fmin,Fmax,Freq,Ti,Ncyc/Freq),...
                    Ti, dt, Ncyc/Freq, MODAL, PLOT, ad2Mass(:,1:2),Freq, VideoName);
ad2U = ad2U.';

end

%% Time integrator
function [T, U, V, A] = Implicit_NL(func, t0, dt, tf, MODAL, PLOT, XY, Freq, VideoName)
if PLOT ~= 0 && ~isempty(VideoName)
    FileName = VideoName;
    %% Create the object
    FPS = round(1/Freq/dt/PLOT); % Frames per second
    writerObj = VideoWriter(FileName,'MPEG-4');
    writerObj.Quality = 100;
    writerObj.FrameRate = FPS; % frames per second
    open(writerObj); 
    figure(1);
    pause;
end

global as1Springs;
% Newmark parameters
gamma = 0.5;
beta  = 0.25;
a0 = 1/(beta*dt^2);
a1 = gamma/(beta*dt);
a2 = 1/(beta*dt);
a3 = 1/(2*beta)-1;
a4 = gamma/beta-1;
a5 = dt*(gamma/(2*beta)-1);

% Time sample
kT = 0;
T = t0:dt:tf; nT = length(T);
up = []; vp = [];
fprintf(1, '\n');
fprintf(1, 'Newmark loop on time steps (%d time intervals)\n', nT);
fprintf(1, '[0%%                                            100%%]\n');
fprintf(1, '[');
STARS = zeros(50,1); star = 0;
global OUTPUT
OUTPUT = 1;
kPlot = 0;
Name{1} = 'Normal';
Name{2} = 'Diagonal';
Name{3} = 'Shear';
for t = t0:dt:tf
    if mod(kT+1,round(nT/50))==0
        star = star + 1;
        if star <= 50 && STARS(star) == 0
            fprintf(1, '*');
            STARS(star) = 1;
        end
    end
    kPlot = kPlot + 1;
    % Calculate internal forces and system matrices
%     up1 = up; vp1 = vp;
%     ad2NonLinearConnection1 = ad2NonLinearConnection;
    output = func(t, up, vp);
    if (kT == 0 || MODAL == 1)
        K = output.K;
        C = output.C;
        M = output.M;
        dof = output.dof;
        if MODAL == 0
            Keq = K + a1*C + a0*M;
            ad2L = chol(Keq(dof,dof), 'lower');
            ad2U = ad2L';
            OUTPUT = 0;
        end
    end
    Fint = output.Fint;
    Fext = output.Fext;
    F = Fext + Fint;

    %% Calculate projection mode
    
    if (kT == 0 && MODAL == 1)
        nDof = length(dof);
        [PHI,W2] = eigs(K(dof,dof),M(dof,dof),min(1000,nDof),'sm');
        W2 = diag(W2);
        W = sqrt(W2);
        Freq = W/2/pi;
        keepModes = find(Freq<1/(dt*5));
        PHI = PHI(:,keepModes);
        nDof = size(K,1);
        Modes = zeros(nDof,size(PHI,2));
        Modes(dof,:) = PHI;
        PHI = Modes;
        clear Modes

        nMode = size(PHI,2);
        nDof = size(K,1);
        up = zeros(nDof,1);
        U = zeros(nDof,length(T));
        V = zeros(nDof,length(T));
        A = zeros(nDof,length(T));
        ump = zeros(nMode,1);
        vmp = zeros(nMode,1);
        amp = zeros(nMode,1);
    elseif kT == 0
        nDof = size(K,1);
        U = zeros(nDof,length(T));
        V = zeros(nDof,length(T));
        A = zeros(nDof,length(T));
        uc = zeros(nDof,1);
        up = zeros(nDof,1);
        vp = zeros(nDof,1);
        ap = zeros(nDof,1);
    end
    
    if MODAL == 1
        % Calculate modal response
        Km = PHI'*K*PHI;
        Cm = PHI'*C*PHI;
        Mm = PHI'*M*PHI;
        Fm = PHI'*F;
        Keq = Km + a1*Cm + a0*Mm;
        dum = Keq\(Fm + Cm*((a4+1)*vmp + a5*amp) + Mm*(a2*vmp + a3*amp));
        umc = ump + dum;
        vmc = a1*dum - (a4+1)*vmp - a5*amp;
        amc = a0*dum - a2*vmp - a3*amp;

        % Calculte modal static response
        dums = Km \ Fm;

        % Calculate static deflection of all modes
        dus = zeros(nDof,1);
        dus(dof) = K(dof,dof)\F(dof);

        % Calculate final displacements
        uc = up + dus + PHI*(dum-dums);
        % Calculate final velocities
        vc = PHI*vmc;
        % Calculate final velocities
        ac = PHI*amc;
    else
        
        uc(dof) = ad2U\(ad2L\(...
                    F(dof) + C(dof,:)*(a1*up+(a4+1)*vp+a5*ap) + ...
                             M(dof,:)*(a0*up+    a2*vp+a3*ap) + ...
                             K(dof,:)*up));
        vc = a1*(uc-up) - a4*vp - a5*ap;
        ac = a0*(uc-up) - a2*vp - a3*ap;
    end
    
    % Store variables
    kT = kT + 1;
    U(:,kT) = uc;
    V(:,kT) = vc;
    A(:,kT) = ac;
    
    % Udate variables
    up = uc;
    vp = vc;
    ap = ac;
    if MODAL
        ump = umc;
        vmp = vmc;
        amp = amc;
    end
    if PLOT && isempty(VideoName)
        ad1X = uc(1:2:end);
        ad1Y = uc(2:2:end);
        plot(XY(:,1)+ad1X,XY(:,2)+ad1Y,'ob');
        xlim([min(XY(:,1))-0.1 max(XY(:,1))+0.1]);
        ylim([min(XY(:,2))-0.1 max(XY(:,2))+0.1]);
        title(['T = ' num2str(t) '; Force = ' num2str(sum(Fext))]);
        drawnow;
    end
    if PLOT ~= 0 && PLOT == kPlot && ~isempty(VideoName)
        for iType = 1: 1: length(Name)
            subplot(1,length(Name),iType);
            kPlot = 0;
            ad1X = uc(1:2:end);
            ad1Y = uc(2:2:end);
            plot(XY(:,1)+ad1X,XY(:,2)+ad1Y,'ob'); hold on;
            for iSpring = 1: 1: length(as1Springs)
                Spring = as1Springs{iSpring}; 
                M1 = Spring.M1;
                M2 = Spring.M2;
                if ~strcmp(Spring.Name,Name{iType}), continue; end
                if strcmp(Spring.Type,'Linear')
                    color = [0 1 0];
                elseif strcmp(Spring.Type,'GapComp')
                    if Spring.Displ < -Spring.gap
                        color = [0 1 0];
                    else
                        color = [1 1 1];
                    end
                elseif strcmp(Spring.Type,'GapTract')
                    if Spring.Displ > Spring.gap
                        color = [0 1 0];
                    else
                        color = [1 1 1];
                    end
                elseif strcmp(Spring.Type,'StickSlip')
                    Fact = abs(Spring.Force/Spring.ForceMax);
                    color = [Fact 1-Fact 0];
                elseif strcmp(Spring.Type,'3StepSpring')
                    Fact = abs(Spring.Force/Spring.FtRef);
                    if Fact > 1, Fact = 1; end
                    color = [Fact 1-Fact 0];
                end
                plot(XY([M1 M2],1)+ad1X([M1 M2]),XY([M1 M2],2)+ad1Y([M1 M2]),'color',color);
            end
            xlim([min(XY(:,1))-0.1 max(XY(:,1))+0.1]);
            ylim([min(XY(:,2))-0.1 max(XY(:,2))+0.1]);
            title([Name{iType} ' springs']);
            drawnow;
            hold off;
        end
        writeVideo(writerObj,getframe(gcf));
    end
end
if PLOT ~= 0 && ~isempty(VideoName)
    close(writerObj);
end
for iStar = 1: 1: length(STARS)
    if STARS(iStar) == 0
        STARS(iStar) = 1;
        fprintf(1, '*');
    end
end
fprintf(1, ']\n');
end

%% Integrating function
function output = ForcesAndMatrices(t,ad1U,ad1V,...
                ad2Mass, LoadedGrain, Fmin, Fmax, Freq, Ti, Tf)
global OUTPUT
% External load
if t < 0
    Fload = Fmin*(Ti-t)/Ti;
elseif t > Tf
    Fload = Fmin+(t-Tf)/Ti*Fmin;
    if Fload < 0, Fload = 0; end
else
    Fload = (Fmin+Fmax)/2 + (-Fmax+Fmin)/2*cos(2*pi*Freq*t);
end
if isempty(ad1U), ad1U = zeros(2*size(ad2Mass,1),1); end
if isempty(ad1V), ad1V = zeros(2*size(ad2Mass,1),1); end
ad1Fext = zeros(2*size(ad2Mass,1),1);
ad1Fext(2*LoadedGrain) = -Fload/length(LoadedGrain);
% Internal forces and system matrices
if OUTPUT
    [ad1Fint, ad2K, ad2C, ad2M] = CalculateForcesAndMatrices(ad1U,ad1V,ad2Mass);
    dof = [ad2Mass(:,4)';ad2Mass(:,5)']; dof = find(dof(:));
    output.Fext = ad1Fext;
    output.Fint = ad1Fint;
    output.K = ad2K;
    output.C = ad2C;
    output.M = ad2M;
    output.dof = dof;
else
    ad1Fint = CalculateForcesAndMatrices(ad1U,ad1V,ad2Mass);
    output.Fext = ad1Fext;
    output.Fint = ad1Fint;
end
% Output
end
